package model;

/* 3D Scanner Control Version 1.0 (c) 2009 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;

import control.Main;

import view.ShowMessage;

public class ScannerData {

	private final int SIZEX = 256;
	private final int SIZEY = 256;
	private final int PVALUES = 6;
	
	public static final String FILE_ENDING = ".3draw";
	
	//configuration: x, y, datapoint: x, y, z, light, size, recording mode 
	private int data[][][];
	private BufferedImage screeni;
	private int drawnwidth[][];
	
	private ImageIcon ii;
	
	private boolean olddrawmode = true;
	private float oldcontrast = 0;
	private final String MACIGFILEID ="3DScannerDataFile-V002";
	
	private String filename;
	
	
	public ScannerData(String filename) {
		data = new int[SIZEX][SIZEY][PVALUES];
		drawnwidth = new int[SIZEX][SIZEY];
		screeni = new BufferedImage(SIZEX, SIZEY, BufferedImage.TYPE_INT_RGB);
		ii = new ImageIcon(screeni);
		loadFile(filename);
		this.filename = filename;
	}

	public ScannerData() {
		data = new int[SIZEX][SIZEY][PVALUES];
		drawnwidth = new int[SIZEX][SIZEY];
		screeni = new BufferedImage(SIZEX, SIZEY, BufferedImage.TYPE_INT_RGB);
		ii = new ImageIcon(screeni);
		filename = "untitled";
	}

	private float detOptBrightness(boolean distance) {
		int maximum = 1;
		for (int y = 0; y < SIZEY; y++) {
			for (int x = 0; x < SIZEX; x++) {
				int value;
				if (distance) {
					value = data[x][y][2];
				} else
					value = data[x][y][3];
				if (value > maximum)
					maximum = value;
			}
		}
		return (256/((float)maximum));
	}
	
	private void updateDataPoint(int x, int y, boolean distance, float contrast) {
		int size = data[x][y][4];
		int value;
		if (distance) {
			value = data[x][y][2];
		} else
			value = data[x][y][3];
		value *= contrast;
		value = Math.min(value, 255);
		int color = value+(value<<8)+(value<<16);
		//System.out.println("Value["+x+"]["+y+"]: "+value+" Size:"+size+" Color: "+color);
		for (int dx = -size/2; dx < (size+1)/2; dx++) {
			for (int dy = -size/2; dy < (size+1); dy++) {
				int ax, ay;
				ax = dx+x;
				ay = dy+y;
				if ((ax < SIZEX) && (ay < SIZEY) && (ax >= 0) && (ay >= 0)) {
					if ((drawnwidth[ax][ay] == 0) || (drawnwidth[ax][ay] >= size)) {
						screeni.setRGB(dx+x, SIZEY-(dy+y)-1, color);
						drawnwidth[ax][ay] = size;
					}
				}
			}
		}
	}
	
	/**
	 * generates a image icon from the data
	 * @param distance true: draw distance values. false: draw light values
	 * @param contrast scaling factor for the values. <= 0 means auto scale.
	 * @return
	 */
	public ImageIcon getGraphic(boolean distance, float contrast) {
		if (contrast <= 0) {
			contrast = detOptBrightness(distance);
		}
		if ((olddrawmode != distance) || (oldcontrast != contrast)) {
			for (int y = 0; y < SIZEY; y++) {
				for (int x = 0; x < SIZEX; x++) {
					drawnwidth[x][y] = 0;
				}
			}
			for (int y = 0; y < SIZEY; y++) {
				for (int x = 0; x < SIZEX; x++) {
					updateDataPoint(x, y, distance, contrast);
				}
			}
		}
		olddrawmode = distance;
		oldcontrast = contrast;
		return ii;
	}

	public void appExit() {
		saveFile("lastExitDump"+FILE_ENDING);
	}

	public void saveFile(String filename) {
    //open the needed streams
    FileOutputStream srcwriter;
    try {
      srcwriter = new FileOutputStream(filename);
    } catch (FileNotFoundException e) {  //we want to create the file anyway, so this will not happen
      e.printStackTrace();
      return;
    }
    OutputStream os;
    try {
    	os = new GZIPOutputStream(srcwriter);
    	ObjectOutputStream oos;
    	oos = new ObjectOutputStream(os);
    	//write to the stream
    	oos.writeUTF(MACIGFILEID);
    	oos.writeInt(SIZEX);
    	oos.writeInt(SIZEY);
    	oos.writeInt(PVALUES);
    	for (int y = 0; y < SIZEY; y++) {
    		for (int x = 0; x < SIZEX; x++) {
    			for (int k = 0; k < PVALUES; k++) {
    				oos.writeInt(data[y][x][k]);
    			}
    		}
    	}
      oos.close();
    } catch (IOException e) {
    	e.printStackTrace();
    	new ShowMessage("Error: Saving failed because of an IOException");
    }
	}
	
	private void loadFile(String filename) {
		FileInputStream srcreader;
    try {
      srcreader = new FileInputStream(new File(filename));
    } catch (FileNotFoundException e) { //the file was selected by the user, so this should not happen very often
    	System.out.println("ScannerData.loadFile: Error: '"+filename+"' could not be found");
    	new ShowMessage("Error: File '"+filename+"' could not be found");
      //e.printStackTrace();
      return;
    }
    InputStream is;
    try {
    	is = new GZIPInputStream(new BufferedInputStream(srcreader));
    	ObjectInputStream ois = new ObjectInputStream(is);
    	String id = ois.readUTF();
    	if (!id.equals(MACIGFILEID)) {
    		new ShowMessage("Error: File format '"+id+"' is not supported. Aborting");
    		ois.close();
    		return;
    	}
    	int sx, sy, vals;
    	sx = ois.readInt();
    	sy = ois.readInt();
    	vals = ois.readInt();
    	if ((sx != SIZEX) || (sy != SIZEY) || (vals != PVALUES)) {
    		new ShowMessage("Error: File format has a wrong geometry. Aborting");
    		ois.close();
    		return;
    	}
    	for (int y = 0; y < SIZEY; y++) {
    		for (int x = 0; x < SIZEX; x++) {
    			for (int k = 0; k < PVALUES; k++) {
    				data[y][x][k] = ois.readInt();
    			}
    		}
    	}
    	ois.close();
    } catch (IOException e) {
    	e.printStackTrace();
    	new ShowMessage("Error: IO Exception occured while reading the input file");
    }
	}
	
	public void setData(int cx, int cy, int size, String answer) {
		if (answer.startsWith("G:"+Main.SCAN_MODE_LINEAR)) { //linear values. example: 'G:1;X:00312'
			int value = 0;
			try { 
				value = Integer.parseInt(answer.substring(6, 11));
				//System.out.println("Setting linear value "+answer.substring(6, 11));
			} catch (NumberFormatException e) {
				System.out.println("ScannerData.setData: Error: Could not extract value from '"+answer+"'");
			}
			data[cx][cy][2] = value;
			data[cx][cy][4] = size;
			data[cx][cy][5] = 2;
		} else
		if (answer.startsWith("G:"+Main.SCAN_MODE_VECTOR)) { //vector values. example: 'G:3;X:-0331;00014;00204;'
			int x = 0, y = 0, z = 0;
			try {		
				x = Integer.parseInt(answer.substring(6, 11));
				y = Integer.parseInt(answer.substring(12, 17));
				z = Integer.parseInt(answer.substring(18, 23));
				//System.out.println("Setting vector values "+answer.substring(6, 11));
			} catch (NumberFormatException e) {
				System.out.println("ScannerData.setData: Error: Could not extract values from '"+answer+"'");
			}
			data[cx][cy][0] = x;
			data[cx][cy][1] = y;
			data[cx][cy][2] = z;
			data[cx][cy][4] = size;
			data[cx][cy][5] = 3;
		} else
		if (answer.startsWith("G:"+Main.SCAN_MODE_LIGHT)) { //light values
			int value = 0;
			try {
				value = Integer.parseInt(answer.substring(6, 11));
				//System.out.println("Setting light value "+answer.substring(6, 11));
			} catch (NumberFormatException e) {
				System.out.println("ScannerData.setData: Error: Could not extract value from '"+answer+"'");
			}
			data[cx][cy][3] = value;
		} else
			System.out.println("ScannerData: Error: '"+answer+"are invalid values");
		updateDataPoint(cx, cy, olddrawmode, oldcontrast);
	}
	
	public void savePNG(File f) {
		RenderedImage ri = screeni;
		try {
			ImageIO.write(ri, "png", f);
		} catch (IOException e) {
			new ShowMessage("Error: Saving PNG as '"+f.getName()+"' failed with an IOException");
			e.printStackTrace();
		}
	}
	
	public String getFilename() {
		return filename;
	}

	public int getYres() {
		return SIZEY;
	}
	
	public int getXres() {
		return SIZEX;
	}

	public int getX(int x, int y) {
		return data[x][y][0];
	}
	
	public int getY(int x, int y) {
		return data[x][y][1];
	}
	
	public int getZ(int x, int y) {
		return data[x][y][2];
	}
	
	public double getBrightness(int x, int y) {
		return data[x][y][3];
	}
	
	public int getResolution(int x, int y) {
		return data[x][y][4];
	}

}
