package control;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;

import model.ScannerData;

import view.ShowMessage;

/* 3D Scanner Control Version 1.0 (c) 2009 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

public class PovRayExport {

	/**
	 * 
	 * @param filename
	 * @param minDist
	 * @param maxDist
	 * @param objectType 1: spheres, 2: cubes
	 * @param textureMode 0: simpleLightGrey 1: brighness from Sensor, 2: wood 
	 * @return true: if success, false if not
	 */
	public static boolean exportFile(String filename, ScannerData sd, int minDist, int maxDist, int objectType, int textureMode) {
		if (minDist >= maxDist) {
			new ShowMessage("Error: minimum distance must be lower than the maximum distance");
			return false;
		}
    try {
    	Writer wf = new BufferedWriter(new FileWriter(new File(filename)));
    	//write header
    	//short comment
    	wf.append("//Pov Ray file generated from 3D Scanner Control 1.0\n");
    	wf.append("//For building the 3D scanner, see www.marwedels.de/malte\n");
    	wf.append("//A value of 1.0 represents 1cm\n");
    	wf.append("\n");
    	//Include various files
    	wf.append("#include \"colors.inc\"\n");
    	wf.append("#include \"textures.inc\"\n");
    	wf.append("#include \"metals.inc\"\n");
    	wf.append("#include \"stones.inc\"\n");
    	wf.append("\n");
    	//Light
    	wf.append("//Setting up a light\n");
    	wf.append("light_source { <20,0,-100> color <1,1,1> }\n");
    	//Camera
    	wf.append("//Setting up a camera\n");
    	wf.append("camera { location <0,0,-75> look_at <0,0,0> }\n");
    	//Plane
    	wf.append("//A infinitive plate\n");
    	wf.append("plane { <0,1,0>, "+getLowestY(sd)+" texture { pigment { checker Black, White }}}\n");
    	wf.append("\n");
    	//The objects
    	wf.append("//The Data from the 3D Scanner\n");
    	wf.append("\n");
    	//Generate objects
    	double maxbright = getMaximumBrighntess(sd);
    	int objectCount = 1; //one for the plane
    	for (int ax = 1; ax < sd.getXres(); ax++) {
    	  for(int ay = 1; ay < sd.getYres(); ay++) {
    	    double  pz = sd.getZ(ax, ay);
    	    //we do not need empty fields or noise
    	    if ((pz >= minDist) && (pz < maxDist)) {
    	    	double px = sd.getX(ax, ay);
    	    	double py = sd.getY(ax, ay);
    	    	//try equalize the y direction as the scanner is not linear here
    	    	//py += Math.abs(px)*0.002*py;
    	      double res = sd.getResolution(ax, ay);
    	      double bright = sd.getBrightness(ax, ay);
    	      double resa = (-1)*res/2; //beginning
    	      double resb = res/2;      //end
    	      bright /= maxbright;     //except 200 is the maximum brightness
    	      //scale from cm to mm
    	      px /= 10;
    	      py /= 10;
    	      pz /= 10;
    	      //now generate object
    	      if (objectType == 2) { //use boxes 
    	      	wf.append("box { <"+resa+","+resa+","+resa+">, <"+resb+","+resb+","+resb+">\n");
    	      } else if (objectType == 1) { //use spheres
    	      	wf.append("sphere {  <1,1,1>, "+res+"\n");
    	      }
    	      wf.append("  translate <"+px+","+py+","+pz+">\n");
    	      if (textureMode == 1) {
    	      	wf.append("  pigment {rgb <"+bright+","+bright+","+ bright+"> }\n");
    	      } else if (textureMode == 2) {
    	      	wf.append("  texture { DMFWood4 }\n");
    	      } else // simple gray (textureMode == 0)
     	      	wf.append("  pigment {rgb <0.8,0.8,0.8> }\n");
    	      wf.append("}\n");
    	      objectCount++;
    	    }
    	  }
    	}
      wf.close();
      System.out.println("PovRayExporter: Exported "+objectCount+" visible objects");
    } catch (IOException e) {  //we want to create the file anyway, so this will not happen
      e.printStackTrace();
      return false;
    }
    return true;
	}
	
	
	private static double getLowestY(ScannerData sd) {
		double miny = -10; //must be always below the camera
		for (int y = 0; y < sd.getYres(); y++) {
			for (int x = 0; x < sd.getXres(); x++) {
				double py = sd.getY(x, y);
				py /= 10;
				miny = Math.min(py, miny);
			}
		}
		return miny;
	}
	
	private static double getMaximumBrighntess(ScannerData sd) {
		double maxb = 0;
		for (int y = 0; y < sd.getYres(); y++) {
			for (int x = 0; x < sd.getXres(); x++) {
				double pb = sd.getBrightness(x, y);
				maxb = Math.max(pb, maxb);
			}
		}
		return maxb;
	}
	

}
