/*Uart Functions C File
  (c) 2005, 2008, 2009 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

//External functions
#include "uart.h"

//0xff marks the buffer as empty, so no rp wp comparison is needed
u08 volatile uart_rx_buf[UART_RX_BUF_SIZE]; //Receive buffer
u08 uart_rx_rp = 0;
u08 volatile uart_rx_wp = 0;

u08 volatile uart_tx_buf[UART_TX_BUF_SIZE]; //Transmit buffer
u08 volatile uart_tx_rp = 0;
u08 uart_tx_wp = 0;
u08 volatile uart_tx_busy = 0; //set to 0 by the int


u08 volatile uart_error;
/* bit 0: rx buff full
   bit 1: frame error
   bit 2: data overrun
   bit 3: 0xff send. this is invalid
   bit 4: internal fifo error
   bit 5: external fifo error
*/

/*
How it works:
Sending and reciving is done by interrupts and a FIFO.
A byte in the FIFO is empty if it has the value 0xff.
By avoiding explicite comparisons between the read and write pointers,
no interrupt disable commands are needed for accessing the FIFO.
Keep in mind, that you should not use the pointers used in the interrups
for calculating free space or avariable data, because doing so may results
in nasty side-effects.
This is there reason why there are the functions uart_rx_available(),
uart_tx_free() and not uart_tx_available(), uart_rx_free().
*/


void uart_init(void) {
	UBRRL = UARTNUMBER;  //set speed
	UCSRC = (1<<URSEL) | (1<<UCSZ1) | (1<<UCSZ0); //set 8Bit mode
	//TX enabled, RX enabled, RX int enabled
	UCSRB = 1<<RXEN | 1<<TXEN | 1<< RXCIE | 1<<TXCIE;
	int i;
	for (i = 0; i < UART_RX_BUF_SIZE; i++) {
		uart_rx_buf[i] = 0xff;
	}
	for (i = 0; i < UART_TX_BUF_SIZE; i++) {
		uart_tx_buf[i] = 0xff;
	}
}

/*
Transmitts a byte from the FIFO.
Note: This function may be called from the interrupt and the main routine.
*/
static void uart_transmit(void) {
	u08 rp_l = uart_tx_rp;
	if (uart_tx_buf[rp_l] != 0xff) {
		UDR = uart_tx_buf[rp_l];
		uart_tx_buf[rp_l] = 0xff;
		rp_l++;
		rp_l %= UART_TX_BUF_SIZE;
		uart_tx_rp = rp_l;
	} else
		uart_tx_busy = 0;
}

/*
Waits until there is space in the FIFO.
If the the UDR is empty and the uart not busy, a first transmit is initiated.
All reaming bytes are then transmitted by the interrupt.
Note: If the FIFO is full, waiting may need some time, if there are multiple
bytes to send, consider to feed your watchdog while waiting, otherwise it
may bite you.
*/
void uart_putchar(u08 c) {
	while (uart_tx_buf[uart_tx_wp] != 0xff);
	uart_tx_buf[uart_tx_wp] = c;
	uart_tx_wp++;
	uart_tx_wp %= UART_TX_BUF_SIZE;
	if (uart_tx_busy == 0) { //if the int wont get new data
		uart_tx_busy = 1;
		while (!(UCSRA & (1<<UDRE))); //wait until there is free space
		uart_transmit();
	}
}

u08 uart_rx_available(void) {
	u08 us = 0;
	u08 p = uart_rx_rp;
	while ((uart_rx_buf[p] != 0xff) && (us < UART_RX_BUF_SIZE)) {
		us++;
		p++;
		p %= UART_RX_BUF_SIZE;
	}
	return us;
}

u08 uart_tx_free(void) {
	u08 f = 0;
	u08 p = uart_tx_wp;
	while ((uart_tx_buf[p] == 0xff) && (f < UART_TX_BUF_SIZE)) {
		f++;
		p++;
		p %= UART_TX_BUF_SIZE;
	}
	return f;
}

// Gets a byte from the FIFO. Returns 0xff if the FIFO is empty
u08 uart_rx_get(void) {
	u08 data = uart_rx_buf[uart_rx_rp];
	if (data != 0xff) {
		uart_rx_buf[uart_rx_rp] = 0xff;
		uart_rx_rp++;
		uart_rx_rp %= UART_RX_BUF_SIZE;
	} else { //look if external error or internal fifo error
		if (uart_rx_available() == 0) {
			uart_error |= 0x10;
		} else
			uart_error |= 0x20;
	}
	return data;
}

ISR(SIG_UART_RECV) {
	while (UCSRA & 1<< RXC) { //there may be two bytes waiting
		if (UCSRA & 1<<FE) { //if frame error
			uart_error |= 0x02;
		}
		if (UCSRA & 1<<DOR) { //if data overrun
			uart_error |= 0x04;
		}
		u08 data = UDR;
		u08 wp_l = uart_rx_wp;
		if (uart_rx_buf[wp_l] == 0xff) {
			if (data != 0xff) {
				uart_rx_buf[wp_l] = data;
				wp_l++;
				wp_l %= UART_RX_BUF_SIZE;
				uart_rx_wp = wp_l;
			} else
				uart_error |= 0x08;
		} else
			uart_error |= 0x01;
	}
}

ISR(SIG_UART_TRANS) {
	uart_transmit();
}

u08 uart_error_get(void) {
	cli();
	u08 tmp = uart_error;
	uart_error = 0;
	sei();
	return tmp;
}
