package control;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;

/* Charge Control Version 1.1 (c) 2008 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/**
 * Provides a simple way to store some configurations.
 * This class implements the singleton pattern.
 */
public class ConfigFile {

	private final String usedFileName = ".chargecontrolcfg";
	private String usedAbsoluteFilePath;
	
	private ArrayList<String> keys;
	private ArrayList<String> values;	
	
	private static ConfigFile cf;
	
	public static ConfigFile getInstance() {
		if (cf == null) {
			cf = new ConfigFile();
		}
		return cf;
	}
	
	private ConfigFile() {
		keys = new ArrayList<String>();
		values = new ArrayList<String>();
		readFromFile(System.getProperty("user.home")+System.getProperty("file.separator")+usedFileName);
	}
	
	/**
	 * Searches the value of the given key
	 * @param key the key where the value should be returned
	 * @return the value, null if the key was not found
	 */
	public String getSettings(String key) {
		int i = getKeyPosition(key);
		if (i >= 0) {
			return values.get(i);
		}
		return null;
	}
	
	/**
	 * Sets a value for the key
	 * @param key May not contain a colon ':' and a '\n' in the key
	 * @param value May not contain a '\n' in the value
	 */
	public void setSettings(String key, String value) {
		int i = getKeyPosition(key);
		if (i >= 0) {
			values.set(i, value);
		} else
			add(key, value);
		saveToFile(usedAbsoluteFilePath);
	}
	
	private void add(String key, String value) {
		keys.add(key);
		values.add(value);
	}
	
	/**
	 * Returns the index of the given key
	 * @param key the key to search
	 * @return Index in array if found, -1 if not found
	 */
	private int getKeyPosition(String key) {
		for (int i = 0; i < keys.size(); i++) {
			if (keys.get(i).equals(key)) {
				return i;
			}
		}
		return -1;
	}
	
	private void saveToFile(String filename) {
		File f = new File(filename);
    FileOutputStream srcwriter;
    try {
      srcwriter = new FileOutputStream(f);
    } catch (FileNotFoundException e) {
      System.out.println("ConfigFile.saveToFile: Hint: Config file '"+filename+"' not found");
      return;
    }
    OutputStream os = new BufferedOutputStream(srcwriter);
    for (int i = 0; i < keys.size(); i++) {
    	String wr = keys.get(i)+":"+values.get(i)+"\n";
    	try {
				os.write(wr.getBytes());
			} catch (IOException e) {
				 System.out.println("ConfigFile.saveToFile: Error: Could not write to outputStream");
		     break;
			}
    }
		try {
			os.close();
		} catch (IOException e) {
			 System.out.println("ConfigFile.saveToFile: Error: Could not close outputStream");
			//who cares... the stream is not needed anyway in the future
		}
	}
	
	private void readFromFile(String filename) {
		File f = new File(filename);
		usedAbsoluteFilePath = f.getAbsolutePath();
		//System.out.println("Config file is: "+usedAbsoluteFilePath);
    FileInputStream srcreader;
    try {
      srcreader = new FileInputStream(f);
    } catch (FileNotFoundException e) {
      System.out.println("ConfigFile.readFromFile: Hint: Config file '"+filename+"' not found");
      return;
    }
    InputStream is = new BufferedInputStream(srcreader);
    try {
			while (is.available() > 0 ) {
			  String line = Tools.extractline(is);
			  int pos = line.indexOf(":");
			  if (pos >= 0) {
			  	add(line.substring(0, pos),line.substring(pos+1));
			  }
			}
		} catch (IOException e) {
      System.out.println("ConfigFile.readFromFile: Waring: Could not properly read from "+filename+"'");
		}
		try {
			is.close();
		} catch (IOException e) {
			//who cares... the stream is not needed anyway in the future
		}
	}
	
}
