package control;

/* Charge Control Version 1.1 (c) 2008 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/**
 * This class implements the singleton pattern.
 * It its managing under which conditions a new Session should be created for the incoming data.
 * Moreover it looks for incoming data and forwards them to the current session.
 * 
 * @author Malte Marwedel
 *
 */


public class SerialToSession {

	private static SerialToSession sts;
	
	/**
	 * The rules for each incoming data line decide
	 * if NO_SESSION -> create a new Session and set state to PROBE
	 * if PROBE and probe data comes in -> do not change state
	 * if PROBE and non probe data comes in -> switch to WORKING state
	 * if WORKING and probe data comes in -> switch to PROBE state and generate new Session
	 * otherwise: do not change state
	 */
	private final int NO_SESSION = 0;
	private final int PROBE = 1;
	private final int WORKING = 2;
	
	private Session activeSession;
	private int state;
	
	/**
	 * For fail-safe reason. Guess there is a program error or wrong serial data if we generate
	 * a few hundred windows making the user desktop unusable 
	 */
	private int wouldCreateSession = 0;
	private final int maxSessions = 30;
	
	public static SerialToSession getInstance() {
		if (sts == null) {
			sts = new SerialToSession();
		}
		return sts;
	}
	
	private SerialToSession() {
		state = NO_SESSION;
	}
	
	/**
	 * Call this function whenever the user close the session, or the serial port
	 *
	 */
	public void stop() {
		state = NO_SESSION;
		activeSession = null; //make RAM freeing possible
	}
	
	public void lookIntoFIFO() {
		SerialThread st = SerialThread.getInstance();
		boolean needsupdate = false;
		while (st.getFifoSize() > 0) {
			String line = st.getFromFifo();
			handleLine(line);
			needsupdate = true;
		}
		if ((activeSession != null) && (needsupdate))
			activeSession.updateGFX();
	}
	
	private void handleLine(String data) {
		if (state == NO_SESSION) {
			wouldCreateSession++;
			if (wouldCreateSession <= maxSessions) { //failsafe
				activeSession = new Session();
				state = PROBE;
			} else {
				//do not use a dialog box, because those could flood the desktop too
				System.out.println("SerialToSession: Warning: Not creating a new Session window,"+
						"because we already created a lot, so this might be a programming error and"+
						"I don't want to flood your desktop with hundreds of Windows you have to click away."+
						"This would have been Session window "+wouldCreateSession);
			}
		} else if (state == PROBE) {
			if (data.startsWith("I:1")) {
				//this is probe data
			} else
				state = WORKING;			
		} else if (state == WORKING) {
			if (data.startsWith("I:1")) {
				//this is probe data
				activeSession.updateGFX(); //tell the old Session that an update should occur
				activeSession = new Session();
				state = PROBE;
			}
		}
		activeSession.addDataLine(data);
	}

	public Session getActiveSession() {
		return activeSession;
	}
	
}
