package data;

/* Charge Control Version 1.1 (c) 2008 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;

import control.Tools;

import model.RecordedData;

public class ChargingData extends UsefulData implements RecordedData {

	//Information send on the beginning
	int chargingcurrent = -1;

	//Information send on the termination
	int charged = -1;


	/**
	 * Expected for bufline:
	 * On starting: B:cc;SSSSS;CCCCC;PP
	 * B: Begins
	 * cc: Cells: how many the battery(pack) has
	 * SSSSS: Speed: the planned charging current in mA.
	 * CCCCC: Capacity of the battery in mAh
	 * PP: Charged percent, how full the battery already is at the beginning.
	 *
	 * null is also allowed, but only not recommend
	 * @param bufline: see above
	 */
	public ChargingData(String bufline) {
		al = new ArrayList<DataEntry>();
		if (bufline != null) {
			if (bufline.length() == 19) {
				cells = Integer.valueOf(bufline.substring(2, 4));
				chargingcurrent = Integer.valueOf(bufline.substring(5, 10));
				capacity = Integer.valueOf(bufline.substring(11, 16));
				sPercent = Integer.valueOf(bufline.substring(17, 19));
			} else
				System.out.println("ChargingData.init: Error: Could not use '"+bufline+"', wrong length");
		}
	}

	/**
	 * Accepted strings:
	 * Strings starting with 'I:2'
	 * On stopping: T:R;MM;HH;CCCCC;PP
	 * T: Termination
	 *   R: Reason for termination:
   *   A: Abort by user
   *   C: charging current was too low (charging only, like battery disconnected)
   *   P: battery is full (100% percent, charging only)
   *   D: Delta/V detection (charging only)
   *   E: Empty (discharging only)
	 * MM: Minutes the process needed
	 * HH: Hours the process needed
	 * CCCCC: How much was charged/discharged in mAh
	 * PP: Percent the battery is full (charging only, will always reported as 0 after
   * a discharge)
	 */
	public boolean add(String str) {
		if (str.startsWith("I:2")) {
			DataEntry de = new DataEntry(str);
			al.add(de);
		} else if (str.startsWith("T:")) { //termination detected
			int strlen = str.length();
			if ((strlen == 18) || (strlen == 19)) {
				reason = str.charAt(2);
				minutes = Integer.valueOf(str.substring(4, 6));
				hours = Integer.valueOf(str.substring(7, 9));
				charged = Integer.valueOf(str.substring(10, 15));
				if (strlen == 18) {	//old format
					if (reason != 'P') {
						tPercent = Integer.valueOf(str.substring(16, 18)); //only has two digits
					} else
						tPercent = 100;
				} else //new format
					tPercent = Integer.valueOf(str.substring(16, 19)); //three digits
			} else {
				System.out.println("ChargingData.add: Error: Could not use '"+str+"', wrong length");
				return false;
			}
		} else {
			System.out.println("ChargingData.add: Error: Could not handle dataline '"+str+"'");
			return false;
		}
		return true;
	}

	public void pushToStream(OutputStream outs) throws IOException {
		if ((cells >= 0) && (chargingcurrent >= 0) && (capacity >= 0) && (sPercent >= 0)) {
			String str = new String("B:"+Tools.intToStr(cells,2)+";"+
					Tools.intToStr(chargingcurrent,5)+";"+Tools.intToStr(capacity, 5)+";"+
					Tools.intToStr(sPercent, 2)+Tools.newline);
			outs.write(str.getBytes());
		}
		for (int i = 0; i < al.size(); i++) {
			outs.write((al.get(i).getAsString()+Tools.newline).getBytes());
		}
		if ((reason != ' ') && (minutes >= 0) && (hours >= 0) && (charged >= 0) && (tPercent >= 0)) {
			String str = new String("T:"+reason+";"+Tools.intToStr(minutes, 2)+";"+
					Tools.intToStr(hours, 2)+";"+Tools.intToStr(charged, 5)+";"+Tools.intToStr(tPercent, 2)+Tools.newline);
			outs.write(str.getBytes());
		}

	}

  public String getDescription() {
    return new String("Charged for "+super.getDuration());
  }

	@Override
	public String getAdvancedDescription() {
		String str = "Charged for "+super.getDuration()+" (time based on the amount of recorded measurements)"+Tools.newline;
		str += "  Datapoints recorded: "+al.size()+Tools.newline;
		if ((cells >= 0) && (chargingcurrent >= 0) && (capacity >= 0) && (sPercent >= 0)) {
			str += "  Beginning charge:"+Tools.newline+"    Cells: "+cells+", Current: "+chargingcurrent+
			"mA, Capacity: "+capacity+"mAh, "+sPercent+"% full"+Tools.newline;
		} else
			str += "    No Beginning data about the charging recorded"+Tools.newline;
		if ((reason != ' ') && (minutes >= 0) && (hours >= 0) && (charged >= 0) && (tPercent >= 0)) {
			str += "  Ending charge by: "+super.getReasonName()+Tools.newline+
			"    Time: "+hours+":"+Tools.intToStr(minutes,2)+"h, Charged: "+charged+"mAh, Battery now is "+tPercent+"% full";
		} else
			str += "    No Ending data about the charging recorded";
		return str;
	}

	public int getBeginningPercent() {
		return sPercent;
	}

	public int getStoppedPercent() {
		return tPercent;
	}

}
