/* clock.c

Copyright (C) 2006-2007 by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "main.h"

u32 volatile msecs;

void clock_init(void) {
//sets up timer2 for the clock
msecs = 0;
TCNT2 = 0;	//reset counter
OCR2A = 125;	//output compare Register
TIMSK2 |= (1<< OCIE2A);	//enable interrupt on compare match
TCCR2A = (1<<WGM21);	//clear Timer on Compare match
TCCR2B |= 0x04; 	//divider 64. OCR2*TCCR2B= 8000. An interrupt every 1ms @ 8MHZ
}

//runs every millisecond
ISR(SIG_OUTPUT_COMPARE2A) {
msecs++;
}

//returns the milliseconds
u32 get_msecs(void) {
u32 msect;
cli();
msect = msecs;
sei();
return msect;
}

//runs sched() until the "delay" in milliseconds is over
void waitms_sched(u16 delay) {
u32 waituntil = get_msecs();
waituntil += delay;
//rs232_println_atomic("blah");
while (waituntil > get_msecs()) {
  //rs232_println_atomic("blub");
  sched();
}
}

//calculates the time in which the delay is over
void timer_setup(u32 *time, u16 delay) {
*time = get_msecs()+((u32)delay);
}

/*Checks if the time is over -> returns '1' and adds the delay.
If it is not 0 is returned.
This has the advantage that there is an accurate timing.
Example: The clock is set up for 1000ms.
But the time will be checked after 1050ms -> the time will be set up
for 2000 and not 2050, resulting in a new time-over in only 950ms.
*/

u08 timer_renew(u32 *time, u16 delay) {
u08 is_over;
if (*time > get_msecs()) {
  is_over = 0;
} else {
  is_over = 1;
  *time += ((u32)delay);
}
return is_over;
}

/*Checks if the time is over. returns '1' and set the clock to the highest end
if it is - otherwise 0. */

u08 timer_check(u32 *time) {
u08 is_over;
if (timer_renew(time, 0)) {
  is_over = 1;
  *time = 0xffffffff;
} else {
  is_over = 0;

}
return is_over;
}

