/* control.c
Reacts on user inputs and sets the values of the charger

Copyright (C) 2007-2008 by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  US
*/

#include "main.h"

u16 maxout_voltage;	//in mV
u16 input_resistor;	//in 10*mOhm

const char selectia[] PROGMEM = "Set current:\nIa:$4mA";

const struct value_selector menu_outputcurrent = {
  selectia, 3, CAL_CHRG_SELECT_MAXCHARGEMILLIAMPS, C_IDX_CHARGINGCURRENT};

const char selectmaxv[] PROGMEM = "Set max voltage:\nUmax:$5mV";

//the maximum selectable voltage will be modified on startup
struct value_selector menu_outputvoltage = {
    selectmaxv, 20, CAL_CHRG_SELECT_MAXV, C_IDX_MAXVOLTAGE};

const char selectcells[] PROGMEM = "Set cells in\nbattery:$2 cells";

//the maximum number of cells will be modified on startup
struct value_selector menu_batterycells = {
     selectcells, 1, 10, C_IDX_CELLS};

const char selectcapacity[] PROGMEM = "Set battery\ncap:$5mAh";

const struct value_selector menu_milliamphours = {
     selectcapacity, 20, CAL_CHRG_SELECT_MAXMILLIAMPHOURS, C_IDX_CAPACITY};

const char selectcharged[] PROGMEM = "Now battery\nis:$3% full";

const struct value_selector menu_chargedpercent = {
     selectcharged, 0, 100, C_IDX_CHARGEPERCENT};


const char checkio[] PROGMEM = "!Checking I/O...";
const char detatchout[] PROGMEM = "!Please detatch\ncharger output";
const char detect_input_stats[] PROGMEM = "!Maxout U:$5mV\nIn R:$50m"OHM_SYMB;
const char thanks[] PROGMEM = "!Thanks";


static void c_detect_input_thanksandwait(void) {
  //a small helper-function for optimization of c_detect_input()
  lcd_print_p(thanks);
  waitms_sched(1500);	//wait after disconnect
  lcd_print_p(checkio);
}

static u08 c_detect_input(void) {
//sets the two global variables input_volatge and input_resistor
//returns 0 if valid, returns 1 if voltage is too low

//checks if output is disconnected
lcd_print_p(checkio);
//sets output to zero
da_probe(0, 10);		// 10mA for removing unwanted currents
waitms_sched(500);
if (get_uppervoltage() > CAL_START_TEST_MAX_UPPER_VOUT) { //if source connected
  //If we have a voltage, then there is a battery connected
  lcd_print_p(detatchout);
  while (get_uppervoltage() > CAL_START_TEST_MAX_UPPER_VOUT) { //200mV
    sched();
  }
  c_detect_input_thanksandwait();
}
//no battery on output, now check input voltage
da_probe(30, 0);
waitms_sched(500);
//no circuits or other things may be connected
if (get_lowervoltage() > CAL_START_TEST_MAX_LOWER_VOUT) { //if sink connected
  lcd_print_p(detatchout);
  while (get_lowervoltage() > CAL_START_TEST_MAX_LOWER_VOUT) { //10mV
    sched();
  }
  c_detect_input_thanksandwait();
}
maxout_voltage = get_uppervoltage();
//check input resistance @ 100mA
//200mA discharging because the amplifier has R2+R3 and not only R3 as normal
da_probe(250, 200);
waitms_sched(1000);
//R = (U1-U2)/I   Since I= 100mA = 0.1A and R in 10*mOhm -> no need to divide
u16 load_voltage = get_uppervoltage();
input_resistor = (maxout_voltage-load_voltage);
da_main();
lcd_printf_p(detect_input_stats, maxout_voltage, input_resistor);
key_wait_until_pressed();
//set max voltage and max-cells in structs
menu_outputvoltage.max = load_voltage - (load_voltage %200); //round to 200mV
menu_batterycells.max = menu_outputvoltage.max/CAL_CELL_MAX_V; //max cells
u08 res = 0;	//avoids the compiler to generate unnecessary returns
if (maxout_voltage < MIN_VOUT) {
  res = 1;
}
return res;
}


const char errorhead[] PROGMEM = "\rError:";
const char seriouserrorhead[] PROGMEM = "\rSerious error:";
const char errortext[CONTROL_MAX_ERRORNUMS+1][16] PROGMEM = {
"Programcode\0",
"Too low Vin (1)\0",
"Heat limiting\0",
"EEPROM invalid\0",
"Too low Vin (2)\0"
};

static void c_error(u08 err) {
/*Shows a error text.
  Numbers above 127 are serious errors which end in an endless loop
*/
da_main();
u08 erri = err & 0x7f;	//extract index number
lcd_clear();
lcd_putchar('\n');
if (erri <= CONTROL_MAX_ERRORNUMS) { //check error index
  lcd_print_p(errortext[erri]);
  rs232_print_number(erri);
} else
  lcd_print_p(errortext[0]); //if out of index -> programming error
if (err & 0x80) {	//a serious errors
  lcd_print_p(seriouserrorhead);
  for (;;) {	//the serious errors -> Endless loop
    sched();
  }
}
lcd_print_p(errorhead);
key_wait_until_pressed();
key_flush();
}

/*Tries to guess how many cells the battery has
during guessing, the battery will be discharged with approx 8 mA
If the battery is empty or there is no battery at all (e.g. for constant
current operation) 0 will be returned */

static void c_detect(void)  {
u08 cells;
u16 outputvoltage;
da_discharge(5);
waitms_sched(500);
outputvoltage = get_outputvoltage();
da_main();
cells = (outputvoltage+CAL_CELL_VERYEMPTY_V/2)/CAL_CELL_LIGHTLOAD_V;
battery_set_cells(cells);
}


const char main_menu[MAIN_MENU_ENTRIES][16] PROGMEM = {
 "Select operation",
 "\nCharge       \0",
 "\nDischarge    \0",
 "\nDis & Charge \0",
 "\nMeasure mAh  \0",
 "\nConst current\0",
 "\nBattery info \0"
};

u08 c_select_main_menu(struct menu_selector * the_menu) {
/* Provides a menu with
  1: Charge
  2: Discharge
  3: Dis & Charge
  4: Measure mAh
  5: Const current
  6: Battery info
  7: Calibration, invisible call by holding the left key
*/
u08 select = 1;
lcd_clear();
lcd_print_p(main_menu[C_M_HEAD]);
key_flush();
while (key_right_pressed() == 0) {
  lcd_print_p(main_menu[select]);
  //key_wait_until_pressed();   //already includes sched()
  if (key_up_pressed()) {
    select--;
  }
  if (key_down_pressed()) {
    select++;
  }
  if (select == 0) {
    select = MAIN_MENU_ENTRIES-1;
  }
  if (select >= MAIN_MENU_ENTRIES) {
    select = 1;
  }
  if (key_left_hold()) { //go to calibration
    return 7;
  }
  sched();
}
//Will be called (nearly) always before going to an other function
c_detect();
return select;
}

/*
  Prints last charged discharged values (if not zero)
  Then  measures the voltage and the voltage with a load for 0.75 seconds.
  The Ri is calculated and displayed with the help of this values if it is below
  655Ohms
*/
const char lastcharged[] PROGMEM =
"!last"BATTERY_SYMB LEFTARROW_SYMB":$5mAh\nlast"BATTERY_SYMB RIGHTARROW_SYMB":$5mAh";
const char pleasewait[] PROGMEM = "!Please wait...\n";
const char uatmamps[] PROGMEM = "U@ $3mA:$5mV\n";
const char ripoints[] PROGMEM = "!Ri:$50m"OHM_SYMB;

u08 c_battery_info(struct menu_selector * the_menu) {
u16 voltage_noload, voltage_load;
u16 test_current;

//Print last charged and discharged if they contain values above zero
if ((battery_get_charged()) || (battery_get_discharged())) {
  lcd_printf_p(lastcharged, battery_get_charged(),battery_get_discharged());
  key_wait_until_pressed();
  if (key_left_pressed())	//no testing, back to menu
    return the_menu->next;
}
lcd_print_p(pleasewait);
da_discharge(0);
waitms_sched(200);
key_flush();		//the 200ms prevents debouching by the way
voltage_noload = get_outputvoltage();
if (voltage_noload == 0) {	//prevent divisions by 0
  voltage_noload = 1;
}
//start with 12mA (think before changing!)
for (test_current = 12; test_current < 800; test_current *= 2) {
  //goes up to 768mA test-discharge
  da_discharge(test_current);
  waitms_sched(750);
  voltage_load = get_outputvoltage();
  u08 percentage = multiplyanddivide(voltage_load, 100, voltage_noload);
  if (percentage < 90) {	//more than 10% lower voltage
    break;			//no further testing
  }
  if (get_current() == 0) { //most likely no battery connected
    break;
  }
  lcd_putchar('*');
}
waitms_sched(750);	//some more time for the output to stop changing
test_current = get_current();	//measure real current
da_main();
lcd_clear();
lcd_printf_p(uatmamps, 0, voltage_noload);
lcd_printf_p(uatmamps, test_current, voltage_load);
key_wait_until_pressed();
//Show Ri: R = U/I; since Ri is in 10mOhm, the multiply with 100 is needed
u16 ri;
ri = multiplyanddivide((voltage_noload-voltage_load), 100, test_current);
if (ri < 0xffff) { //all other Ri are simply not real for batteries
  lcd_printf_p(ripoints, ri, 0);
  key_wait_until_pressed();
}
return the_menu->next;
}


#define SELECTOR_STEPS_C 6
const u16 selector_steps[SELECTOR_STEPS_C] PROGMEM =
                         {100, 200, 500, 1000, 3000, 8000};
const u08 selector_speed[SELECTOR_STEPS_C] PROGMEM =
                         {  5,  10,  20,   50,  100,  200};

static u08 c_selectorsteps_below(u16 value) {
//returns the next decrease step-size on a given input value
u08 k, speedfactor = 1;
for (k = 0; k < SELECTOR_STEPS_C; k++) {
  if (value > pgm_read_word(&selector_steps[k]))
    speedfactor = pgm_read_byte(&selector_speed[k]);
}
return speedfactor;
}

static u08 c_selectorsteps_above(u16 value) {
//returns the next increase step-size on a given input value
return c_selectorsteps_below(value+1);
}

u08 c_value_selection(struct menu_selector * the_menu) {
//this represents a complex selection menu item for a numeric value
const struct value_selector * menu_value = the_menu->val_sel;
u16 selectedvalue = battery_getby_index(menu_value->returnidx);
u08 hold_delay = 0;
//fit value to selector_speed. eg: 1003 -> 1000
selectedvalue = max(menu_value->min,
            selectedvalue-(selectedvalue%c_selectorsteps_below(selectedvalue)));
lcd_clear();
while (key_right_pressed() == 0) {
  if ((key_up_pressed()) || (key_up_hold())) {
    selectedvalue += c_selectorsteps_above(selectedvalue);
    hold_delay = 1;
  }
  if ((key_down_pressed()) || (key_down_hold())) {
    selectedvalue -= c_selectorsteps_below(selectedvalue);
    hold_delay = 1;
  }
  if (key_left_pressed()) { //abort
    return the_menu->prev;
  }
  if ((selectedvalue >= 0x9000) || (selectedvalue < menu_value->min)) {
    selectedvalue = menu_value->min;
  }
  selectedvalue = min(selectedvalue, menu_value->max);
  lcd_putchar('\r');
  lcd_printf_p(menu_value->firstline, selectedvalue, 0);
  if (hold_delay) {
    waitms_sched(100);
    hold_delay = 0;
  }
  sched();
}
battery_setby_index(menu_value->returnidx, selectedvalue);
return the_menu->next;
}

const char outputisu[] PROGMEM = "!Output is:\nU:00.0V I:$4mA";

//offon will be used by c_deltav_onoff() too
const char offon[2][5] PROGMEM = {{'O', 'f', 'f', 0},
                                  {'O', 'n', ' ', 0}};
const char voutmask[] PROGMEM = {0x90+2, '$', '2', 0x90+5, '$', '1', 0};

u08 c_currentnow(struct menu_selector * the_menu) {
lcd_print_p(outputisu);
u08 outputstate = 0;
while (key_right_pressed() == 0) {
  lcd_putchar(0x80+11);
  lcd_print_p(offon[outputstate]);
  if (key_up_pressed()) {	//enable output
    outputstate = 1;
  }
  if (key_down_pressed()) {	//disable output
    outputstate = 0;
  }
  if (key_left_pressed()) {	//select new values
    da_main();
    return the_menu->prev;
  }
  //print current
  lcd_putchar(0x90 + 10);
  lcd_print_word(get_current(),4);
  //switch off on too high voltage
  u16 vout = get_outputvoltage();
  if (vout > battery_get_maxv()) { //if too high voltage
    outputstate = 0;
  }
  //observe internal generated heat
  u16 maxpower = multiplyanddivide(get_current(),
                (maxout_voltage+CAL_VCC_DROP)-vout, 1000);
  if ((outputstate) && (maxpower > MAX_HEAT_POWER)) { //if too much heat
    outputstate = 0;
    da_main();
    c_error(2);
    lcd_print_p(outputisu);
  }
  //print voltage
  lcd_printf_p(voutmask, vout/1000, vout/100);
  if (outputstate == 1) {
    da_charge(battery_get_chargingcurrent());
  } else {
    da_main();
  }
  waitms_sched(20);
}
da_main();
return the_menu->next;
}

u08 c_clear_chargedpercent(struct menu_selector * the_menu) {
battery_set_chargedpercent(0);
return the_menu->next;
}


const char chspeed[] PROGMEM = "!Ch. speed:\nI:0000mA T:";
const char speedmask[] PROGMEM = {(0x90+11),'$' ,'2' ,':' ,'$', '2' ,
                                  (0x90+2), 0};

const char chspeednames[SPEEDGRADES][7] PROGMEM = {
 "slow  ",
 "medium",
 "fast  "
};

u08 c_chargespeed_selection(struct menu_selector * the_menu) {
u08 sp_selection = 0;
u08 sp_selection_o = 1;
u16 chargemilliamps = 1;
u16 chargeminutes = 0;

lcd_print_p(chspeed);
while (key_right_pressed() == 0) {
  if ((key_up_pressed()) && (sp_selection < (SPEEDGRADES-1))) {
    sp_selection++;
  }
  if ((key_down_pressed()) && (sp_selection > 0)) {
    sp_selection--;
  }
  if (key_left_pressed()) { //abort
    return the_menu->prev;
  }
  if (sp_selection != sp_selection_o) {
    lcd_putchar(0x80+10);
    lcd_print_p(chspeednames[sp_selection]);
    chargemilliamps = multiplyanddivide(battery_get_capacity(), 10,
                                        speedfactor[sp_selection]);
    chargemilliamps = min(chargemilliamps,  menu_outputcurrent.max);
    /*BUG: A wrong overchargefactor[] is used even if the chargemilliamps are
      lower because of the menu_outputcurrent.max limit. The expected charging
      time is then calculated wrong and the battery is charged below 100%
      This apply for batteries with a (high) capacity.
      If the maximum charging current is 2.5A:
      over 7.5Ah when selecting the speed 'fast'
      over 13.25Ah when selecting the speed 'medium' or 'slow'
      over 25Ah in all charge speeds, but setting the battery cap is limited
           to 20Ah.
    */
    chargeminutes = multiplyanddivide(battery_get_capacity(),
                               overchargefactor[sp_selection], chargemilliamps);
    chargeminutes = multiplyanddivide(chargeminutes,
                               100-battery_get_chargedpercent(), 100);
    udiv_t displaytime = udiv(chargeminutes, 60);
    lcd_printf_p(speedmask, displaytime.quot, displaytime.rem);
    lcd_print_word(chargemilliamps, 4);
  }
  sp_selection_o = sp_selection;
  sched();
}
battery_set_chargingcurrent(chargemilliamps);
battery_set_overchargefactor(overchargefactor[sp_selection]);
battery_set_speedfactor(speedfactor[sp_selection]);
return the_menu->next;
}

const char deltav[] PROGMEM = "!Stopping by -"DELTA_SYMB"V \ndetection is:";

u08 c_deltav_onoff(struct menu_selector * the_menu) {
/* Let the user select if he wants the -deltaV detection enabled or not.
If the user has selected a fast charge, it is always enabled and this menu
is not displayed */
u08 mode = 1; //1= on, 0=off
u08 mode_o = 0; //results in a first printing on the LCD
lcd_print_p(deltav);
if (battery_get_speedfactor() > SPEED_MEDIUMFAST_THRESOLD) {
  //if a slow or normal charge
  while (key_right_pressed() == 0) {
    if ((key_up_pressed()) && (mode == 0)) {
      mode++;
    }
    if ((key_down_pressed()) && (mode == 1)) {
      mode--;
    }
    if (key_left_pressed()) { //abort
      return the_menu->prev;
    }
    if (mode != mode_o) {
      lcd_putchar(0x80+29);
      lcd_print_p(offon[mode]);
    }
    mode_o = mode;
    sched();
  }
}
battery_set_deltavuse(mode);
return the_menu->next;
}

const char calmask[] PROGMEM = "\rImA: Uup:$5mV\x94 Ulo:$5mV";
const char cal160out[] PROGMEM = "\n+$3";
const char cal160in[] PROGMEM = "\n-$3";
const char caloff[] PROGMEM = "\n 0  ";

u08 c_calibration(struct menu_selector * the_menu) {
key_flush();
while (key_right_pressed() == 0) {
  lcd_printf_p(calmask, get_hiresuppervoltage(), get_hireslowervoltage());
  if ((key_up_hold()) && (!key_down_hold())) {
    if (key_left_hold()) { //700mA charge if both keys are pressed
      da_charge(700);
    } else
      da_charge(160);
    lcd_printf_p(cal160out, get_current(), 0);
  } else
  if ((!key_up_hold()) && (key_down_hold())) {
    da_discharge(160);
    lcd_printf_p(cal160in, get_current(), 0);
  } else {
    da_main();
    lcd_print_p(caloff);
  }
  sched();
}
da_main();
return the_menu->next;
}

struct menu_selector the_menu[C_NUM_MENUPARTS] PROGMEM = {
   {0, 0, NULL, c_select_main_menu},
   {10, 0, &menu_batterycells, c_value_selection},	//1
   {15, 0, &menu_batterycells, c_value_selection},
   {17, 0, &menu_batterycells, c_value_selection},	//3
   {23, 0, &menu_batterycells, c_value_selection},
   {8, 0, &menu_outputcurrent, c_value_selection},	//5
   //battery info
   {0, 0, NULL, c_battery_info},
   //calibration display
   {0, 0, NULL, c_calibration},	//7
   //constant current
   {9, 5, &menu_outputvoltage, c_value_selection},	//8
   {0, 5, NULL, c_currentnow},				//9
   //charging
   {11, 1, &menu_milliamphours, c_value_selection},
   {12, 10, &menu_chargedpercent, c_value_selection},	//11
   {13, 11, NULL, c_chargespeed_selection},
   {14, 12, NULL, c_deltav_onoff},
   {0, 0, NULL, c_chargenow},				//14
   //discharging
   {16, 2, &menu_milliamphours, c_value_selection},
   {0, 0, NULL, c_dischargenow},			//16
   //dis and charge
   {18, 3, &menu_milliamphours, c_value_selection},
   {19, 3, NULL, c_clear_chargedpercent},		//18
   {20, 17, NULL, c_chargespeed_selection},		//19
   {21, 19, NULL, c_deltav_onoff}, //20
   {22, 0, NULL, c_dischargenow},
   {0, 0, NULL, c_chargenow},				//22
   //measure milliamps
   {24, 4, &menu_milliamphours, c_value_selection},
   {25, 4, NULL, c_clear_chargedpercent},		//24
   {26, 23, NULL, c_chargespeed_selection},		//25
   {27, 25, NULL, c_deltav_onoff}, //26
   {28, 0, NULL, c_dischargenow},
   {29, 0, NULL, c_chargenow},				//28
   {30, 0, NULL, c_dischargenow},
   {0, 0, NULL, c_chargenow},				//30
};


const char powerup[] PROGMEM = "!Power up...";
const char showvin[] PROGMEM = "\nU5V=$5mV";
const char eepromdefault[] PROGMEM = "!EEPROM set to\ndefault. Reset";
const char stateconfig[] PROGMEM = {
    33, 40, 99, 41, 50, 48, 48, 54, 45, 50, 48, 48, 56, 44, 71, 80, 76, 10, 77,
    97, 108, 116, 101, 32, 77, 97, 114, 119, 101, 100, 101, 108, 0};

void control_thread(void) {
static struct menu_selector the_menupart;//again static is just for optimization
//set up some default values (a multiple of 200 is recommend)
battery_set_capacity(CAL_DEFAULT_CAPACITY);
battery_set_chargedpercent(0);
battery_set_maxv(DEFAULT_MAX_VOLTAGE); // 1000mV
battery_set_chargingcurrent(DEFAULT_CHARGING_CURRENT); // 100mA
lcd_print_p(powerup);
waitms_sched(500); //to give the other thread time to measure vccvoltage first
/*the waiting above is needed because calling this function only makes sense
after all threads have been run at least once */
#if STACKSTARTDUMP
valid_stack_start_dump();
#endif
if (key_up_pressed() && key_down_pressed()) { //restore default config
  eep_defaulconfig();
  lcd_print_p(eepromdefault);
  while(key_right_pressed() == 0) { //wait for reset
    sched();
    if (key_left_pressed()) //some info about the state of the configuration
      lcd_print_p(stateconfig);
  }
  reset();
}
if (configvalid == 0) { //the variable has to be set-up before scheduling stared
  //There are eeprom read errors
  c_error(3 | 0x80);
}
if (get_vccvoltage() < MIN_VCC) {
  lcd_printf_p(showvin, get_vccvoltage(), 0);
  waitms_sched(2000);
  c_error(1 | 0x80); //Too low Vin (1)
}
if (c_detect_input()) {
  c_error(4 | 0x80); //Too low Vin (2)
}
u08 m_main = 0;
for (;;) {
  if (m_main < C_NUM_MENUPARTS) {
    key_flush();
    memcpy_P(&the_menupart,&the_menu[m_main],sizeof(struct menu_selector));
    m_main = the_menupart.execute(&the_menupart);
  } else
    c_error(0 | 0x80); //Programming error
  sched();
}
}

