/* da_output.c
Everything which has to do with D/A output

Copyright (C) 2006-2007 by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  US
*/


#include "main.h"

u16 volatile charging_current;
u16 volatile discharging_current;

/* Operating mode (op_mode):
0: no output
1: probing output (charge and discharge at the same time possible)
2: charging
3: discharging
*/
u08 volatile op_mode;

static void da_setvalues(u16 ch_current, u16 disch_current, u08 mode) {
charging_current = ch_current;
discharging_current = disch_current;
op_mode = mode;
}

void da_charge(u16 current) {
da_setvalues(current,0, OP_CHARGE);
}

void da_discharge(u16 current) {
da_setvalues(0,current+CAL_CHARGE_OFFSET, OP_DISCHARGE);
}

void da_main(void) {
da_setvalues(0,0, OP_MAIN);
}

void da_probe(u16 ch_current, u16 disch_current) {
da_setvalues(ch_current, disch_current, OP_PROBE);
}

void da_control_thread(void) {
/* Note: Writing most of the following registers uses the tmp register.
   In the case there will be interrupts which use temp, those have to be disabled
   before writing to the registers here.
*/
TCNT1 = 0;		//clear counter
ICR1 = charger_calib[CALIB_L7805VOUT];	//variable PWM resolution
OCR1A = 0;		//lowest value
OCR1B = 0;		//lowest value
TCCR1A = (1<<COM1A1)|(1<<COM1B1)| //enable PWM on OC1A and OC1B
         (1<<WGM11);	//phase correct with ICR1 as top (together with TCCR1B)
TCCR1B = (1<<WGM13)|	//phase correct with ICR1 as top (together with TCCR1A)
         (1<<CS10);	//clock without prescaler

for(;;) {
  //local variables for faster access
  u16 charging_current_l = charging_current;
  u16 discharging_current_l = discharging_current;
  u08 op_mode_l = op_mode;
  //the charging pin
  /*For simplicity we calculate with linear values, even if the output is
    may be slightly non-linear
    Resistor: 1Ohm resulting in 1A each 1V. Vmax: 5V (CAL_L7805_VOUT)
    Note, that CAL_L7805_VOUT = PWM_MAX_VAL.
    Ia = 5V*OCR1A/PWM_MAX_VAL
    <=> OCR1A = Ia*PWM_MAX_VAL/5V
    Note that Ia is in mA => OCR1A = Ia*5000/5000 => OCR1A = Ia
    Since we adapt PWM_MAX_VAL to the AVR Vcc voltage and the Resistor is
    1.0 Ohm, there is no need to multiply or divide :-)
    */
  if ((charging_current_l < CAL_CHRG_SELECT_MAXCHARGEMILLIAMPS) &&
      ((op_mode_l == OP_PROBE) || (op_mode_l == OP_CHARGE))) {
    OCR1A = charging_current_l;
  } else
    OCR1A = 0;		//switch off
  //the discharging pin
  /*For simplicity we calculate with linear values, even if the output is
    may be slightly non-linear
    Resistor: 3.3Ohm resulting in 0.303A each 1V. Vmax: 5V (CAL_L7805_VOUT)
    Note, that CAL_L7805_VOUT = PWM_MAX_VAL.
    Ia = 5V*0.303A*OCR1B/PWM_MAX_VAL
    <=> OCR1B = Ia*PWM_MAX_VAL/(5V*0.303A)
    <=> OCR1B = Ia*5000/(5000mV*0.303A) = Ia/(0.303mVA) = Ia*1000/303
    Note that Ia is in mA => OCR1B = Ia*1000/(303)
  */
  if (discharging_current_l < CAL_CHRG_SELECT_MAXDSMILLIAMPS) {
    if ((discharging_current_l < charger_calib[CALIB_L7805VOUT]) &&
      ((op_mode_l == OP_PROBE) || (op_mode_l == OP_DISCHARGE))) {
      OCR1B = discharging_current_l;
    } else
      OCR1B = 0;	//switch off
  } else
    OCR1B = 0;		//switch off
  //control the LED
  if (op_mode_l == OP_MAIN) {	//LED is off if the output is off too
    led_off();
  } else
    led_on();
  sched();
}
}

