/* user_in.c

Copyright (C) 2007 by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  US
*/

#include "main.h"

/* The lowest bit is the hold state
   The second bit is the pressed state
   key_states[0]: down
   key_states[1]: up
   key_states[2]: left
   key_states[3]: right
*/
u08 volatile key_states[KEY_NUMS];

const u08 key_connections[KEY_NUMS] =
                    {KEY_DOWN_PIN, KEY_UP_PIN, KEY_LEFT_PIN, KEY_RIGHT_PIN};

void keys_thread(void) {
//the predefined values are for easy recognizing in debugging only
static u32 key_pressed_times[KEY_NUMS] = {0xfafafafa,0xfbfbfbfb,
                                          0xfcfcfcfc,0xfdfdfdfd};
u08 input_prev_states = 0xff;	//this means all keys were released before
u08 input_curr_states;
for(;;) {
  input_curr_states = PINC;	//read in the key states
  u08 k;
  for (k = 0; k < KEY_NUMS; k++) {	//for every key do...
    /* Note that the keys set the input to zero - inverted keys.
       If previous state released '1' and new state pressed '0',
         set the 'pressed' bit and update the key_pressed_times[k].
       If the new state is released -> reset the 'hold' bit.
       If the new state is pressed and the current time is greater than
         key_pressed_times[k]+KEY_HOLD_DELAY then set the 'hold' bit.
    */
    u08 key_cs = input_curr_states & key_connections[k];
    u08 key_ps = input_prev_states & key_connections[k];
    u08 key_state_l = key_states[k]; //for optimization
    if (key_cs == 0) {		//if current = pressed
      if (key_ps) {		//if previous = released
        key_state_l |= 0x02;	//set 'pressed' bit
        timer_setup(&key_pressed_times[k], KEY_HOLD_DELAY); //setup delay
      } else			//the 'else' is for optimization only
      if (timer_check(&key_pressed_times[k])) {	//check if delay is over
        key_state_l |= 0x01;	//set 'hold' bit
      }
    } else {			//if current = released
      key_state_l &= ~0x01;	//reset 'hold' bit
    }
    key_states[k] = key_state_l; //write new state back
    if ((key_ps == 0) && (key_cs)) {
      //if previous = pressed and current = released
      waitms_sched(250);	//prevent recognizing the bumping of keys
    }
  }				//end of loop for keys
  input_prev_states = input_curr_states;
  sched();
}
}

void key_wait_until_pressed(void) {
/*returns in the case a key is pressed. at first, all keys are flushed
  but later pressed states are not changed, so key_xy_pressed()  will work
  as expected after this function has retuned.
*/
u08 pr = 0;
key_flush();
while (pr == 0) {
  u08 k;
  for (k = 0; k < KEY_NUMS; k++) {
    u08 key_state_l = key_states[k] & 0x02;
    if (key_state_l) {
      pr = 1;
    }
  }
  sched();
}
}

void key_flush(void) {
/*Sets the 'pressed' state of all keys to 'no' by calling all key_xy_pressed()
  functions. */
key_down_pressed();
key_up_pressed();
key_left_pressed();
key_right_pressed();
}

/*
The key_xy_pressed() functions return 1 in the case the user had
pressed the key xy since the last call of this function.
Otherwise the function returns 0.

The key_xp_hold() function returns 1 in the case the user presses this key for
a while and is still pressing it.
Otherwise this function returns 0
*/

u08 key_down_pressed(void) {
u08 key_down_state_l = key_states[0];
/*Using temp for '&' instead of doing this in the if statement saves 12 bytes
  of code.
  Using the result variable instead of two returns saves additional 6 bytes
  of code.
*/
u08 result = 0;
u08 temp = key_down_state_l & 0x02;	//because of better optimized code
if (temp) {
  key_states[0] = key_down_state_l & ~(0x02);
  result = 1;
}
return result;
}

u08 key_down_hold(void) {
return (key_states[0] & 1);
}

u08 key_up_pressed(void) {
u08 key_up_state_l = key_states[1];
u08 result = 0;
u08 temp = key_up_state_l & 0x02;	//because of better optimized code
if (temp) {
  key_states[1] = key_up_state_l & ~(0x02);
  result = 1;
}
return result;
}

u08 key_up_hold(void) {
return (key_states[1] & 1);
}

u08 key_left_pressed(void) {
u08 key_left_state_l = key_states[2];
u08 result = 0;
u08 temp = key_left_state_l & 0x02;	//because of better optimized code
if (temp) {
  key_states[2] = key_left_state_l & ~(0x02);
  result = 1;
}
return result;
}

u08 key_left_hold(void) {
return (key_states[2] & 1);
}

u08 key_right_pressed(void) {
u08 key_right_state_l = key_states[3];
u08 result = 0;
u08 temp = key_right_state_l & 0x02;	//because of better optimized code
if (temp) {
  key_states[3] = key_right_state_l & ~(0x02);
  result = 1;
}
return result;
}

u08 key_right_hold(void) {
return (key_states[3] & 1);
}

