/* validator.c

Copyright (C) 2006-2008 by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "main.h"

void valid_prepare(u16 depth) {
/* Before stating the multithreading part, call this function to put a special
   pattern onto the stack. This pattern will be tested to check if the stacks of
   the various threads overlap,
*/
unsigned char p1 = 0xaf;
unsigned char p2 = 0xfe;
u16 count;
schedlist_s[0] = SP-depth; //mark end of the stack
/*Possible problem: between the upper and lower lines, the compiler
may not place additional (up to two would be OK, see below) pop commands.
schedlist_s[0] could be decreased by two because the software looks for the two
pattern bytes always above (the stack grows to the lower address) the given
address.
*/
depth >>= 1;		//because we push two times in the loop
for (count = 0; count < depth; count++) {
  asm volatile ("push %0": : "r" (p1));
  asm volatile ("push %0": : "r" (p2));
}
for (count = 0; count < depth; count++) {
  asm volatile ("pop %0": "=r" (p2) : );
  asm volatile ("pop %0": "=r" (p1) : );
}
}


#if SRAMDUMP

/*Prints a copy of the whole SRAM over the UART. Use for debugging only, since
  every printing fully hangs the program for about two seconds.
*/

//NOTE: ATMEGA8 and ATMEGA168 have the SRAM at different address locations
#if RAMEND != 0x4ff
  #warning The target AVR does have other RAM locations than the ATMEGA168 has.
  #warning If an old version of avr-libc is used, this may be a bug in the lib.
#endif

void valid_sram_dump(void) {
u08 * k;
u08 val;
while (1) {	//not that elegant, but produces smaller code
  if (UCSR0A & (1<<UDRE0)) break;
}
UDR0 = '\n';
for (k = ((u08*)RAMEND); k > ((u08*)0x100); k--) {
  val = *k;
  while (1) {	//not that elegant, but produces smaller code
    if (UCSR0A & (1<<UDRE0)) break;
  }
  UDR0 = bintohex(val>>4);
  while (1) {	//not that elegant, but produces smaller code
    if (UCSR0A & (1<<UDRE0)) break;
  }
  UDR0 = bintohex(val & 0x0f);
  asm volatile ("wdr");			//resets the watchdog
}
}

#endif

#if STACKSTARTDUMP

/*Prints a copy of the stack starting addresses over the UART.
  Use for debugging only.
  Should not be called in the validator thread, because
  the using of the uart would let other threads come next and the
  watchdog wont get reset.
*/

void valid_stack_start_dump(void) {
u08 k;
for (k = 0; k < SCHED_THREADS; k++) {
  rs232_print_number(schedlist_s[k]);
}
}

#endif


#if THREADBENCHMARK

#warning Make sure watchdog timeout and stack are big enough.
//for debugging, note increase watchdog timeout while using
//and possible the extra reserved stack for this thread.
static void valid_thread_switch_benchmark(void) {
//measures the longest time which all threads need for switching, prints the
//result every second
static u32 lasttime = 0;	//µs (can only count the first hour)
static u32 maxdelta = 0;	//µs
static u32 timer = 1000;	//for printing every second
cli();
u08 temp = TCNT2;		//maximum 1000/8 = 125
u32 thistime = get_msecs()*1000;	//will do a sei() automatically
thistime += temp*8;
u32 delta = 0;
if (thistime >= lasttime) {
  delta = thistime-lasttime;
}
if (delta > maxdelta) {
  maxdelta = delta;
}
if (timer_renew(&timer, 1000)) {
  rs232_print_number_u32(maxdelta);
  //because the rs232 needs multiple sched() time which is not measured
  maxdelta = 0;
  cli();
  temp = TCNT2;			//maximum 1000/8 = 125
  thistime = get_msecs()*1000;	//will do a sei() automatically
  thistime += temp*8;
}
lasttime = thistime;
}

#endif

void valid_thread(void) {
/* This thread checks for stack overflows and if this is all valid,
   he resets the watchdog.
*/
u08 b1, b2;
u16 ps;
u08 *t;
u08 k;
#if SRAMDUMP
u16 l = 1;
#endif
for(;;) {

  //Use for debugging only
  #if SRAMDUMP
  if (l == 0)		//print every 2^16 th time a dump of the ram
    valid_sram_dump();
  l++;
  #endif
  /*The first thread can not be overwritten, moreover the field for
    schedlist_s[0] is not initialized by default, so it points to the ending
    of the initialized pattern, checking if stack and data overlap
. */
  for (k = 0; k < SCHED_THREADS; k++) {
    ps = schedlist_s[k]; //get stack pointer where it was at thread init
    if (ps > 0) {		//the pointer should be initialized
      t = (u08*)(ps+1);
      b1 = *(t);
      t = (u08*)(ps+2);
      b2 = *(t);
      if (((b1 != 0xaf) || (b2 != 0xfe)) && ((b1 != 0xfe) || (b2 != 0xaf))) {
        //pattern not found -> stack overflow
        /* Beginning the etext with \r\n makes sure that the message
           appears in a new line and is always recognisable as this
           by the PC software. */
        static u08 etext[10] = "\r\nE: S   ";
        /* Note the threadnumber which is shown is the one who get crashed,
           meaning, the previous thread was responsible for the crash.
           And if the number is 00, this means that the thread with the highest
           number wrote beyond the range initialised by valid_prepare(). */
        wordtostr(etext, k, 2, 7);
        //prevent error message on reboot and tells the println not to schedule
        sched_thread = WAS_STACK;
        rs232_println_atomic(etext);
        #if SRAMDUMP
        valid_sram_dump();
        #endif
        for (;;) {
        }
      }
    }
  }
  asm volatile ("wdr");			//resets the watchdog
  #if THREADBENCHMARK
  valid_thread_switch_benchmark();
  #endif
  sched();
}
}

