package control;

/* Charge Control Version 1.1 (c) 2008 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

import model.ReceiveMessage;
import view.CalibrationWindow;
import view.ChooseFile;
import view.ShowMessage;

/**
 * This class implements the singleton pattern
 */
public class Calibration implements ReceiveMessage {

	private static Calibration cal;
	private CalibrationWindow cw;
	public final static int CALIB_ENTRIES = 10; //see calibration_config.h
	public final static int CALIB_HEXBYTES_RW = CALIB_ENTRIES*8;
	
	private final static String FILE_IDENTIFIER = "Charger Calibration data v1\n";
		
	private int[] dataarray; //see calibration_config.c 
	
	
	public static Calibration getInstance() {
		if (cal == null) {
			cal = new Calibration();
		}
		return cal;
	}
	
	private Calibration() {
		dataarray = new int[CALIB_ENTRIES*2];
		cw = new CalibrationWindow(this);
		setDefault();
		setGuiValues();
	}

	public void messageTell(String str) {
		if (str.equals("default")) {
			setDefault();
			setGuiValues();
		} else if (str.equals("reset")) {
			SerialThread.getInstance().sendReset();
		} else if (str.equals("read")) {
			readConfig();
		} else if (str.equals("write")) {
			writeConfig();
		} else if (str.equals("save")) {
			saveConfig();
		} else if (str.equals("open")) {
			openConfig();
		} else if (str.equals("labelupdate")) {
			readValuesFromGui();
		}
	}

	public void show() {
		cw.show();
	}
	
	private void setGuiValues() {
		cw.setEdval1Text(String.valueOf((double)((double)dataarray[0])/((double)dataarray[1])));
		cw.setEdval2Text(String.valueOf((double)((double)dataarray[2])/((double)dataarray[3])));
		cw.setEdval3Text(String.valueOf(1/((double)((double)dataarray[4])/((double)dataarray[5]))));
		cw.setEdval4Text(String.valueOf(1/((double)((double)dataarray[6])/((double)dataarray[7]))));
		cw.setEdval5Text(String.valueOf(((double)dataarray[8])/1000));
		setGuiLabelValues();
	}
	
	private void setGuiLabelValues() {
		cw.setLval1Text(String.valueOf((double)((double)dataarray[0])/((double)dataarray[1])));
		cw.setLval2Text(String.valueOf((double)((double)dataarray[2])/((double)dataarray[3])));
		cw.setLval3Text(String.valueOf(1/((double)((double)dataarray[4])/((double)dataarray[5])))); //R2^-1
		cw.setLval4Text(String.valueOf(1/((double)((double)dataarray[6])/((double)dataarray[7]))));
		cw.setLval5Text(String.valueOf(((double)dataarray[8])/1000));
	}
	
	private void setDefault() {
		dataarray[0] = 1250;	//Uppersum
		dataarray[1] = 4092;	//Uppersum
		dataarray[2] = 625;	  //Lowersum
		dataarray[3] = 8184;	//Lowersum
		dataarray[4] = 1; 		//R2^-1
		dataarray[5] = 1; 		//R2^-1
		dataarray[6] = 1; 		//R3^-1
		dataarray[7] = 1;		  //R3^-1
		dataarray[8] = 4950;	//L7805_VOUT
		dataarray[9] = 1;		  //reserved
		applyInverted();
	}
	
	private void applyInverted() {
		for (int i = 0; i <  CALIB_ENTRIES; i++) {
			dataarray[i+CALIB_ENTRIES] = ~dataarray[i];
		}
	}
	
	public void enableSerialComm() {
		cw.enableSerialComm();
	}
	
	public void disableSerialComm() {
		cw.disableSerialComm();
	}
	
	private void readConfig() {
		String cfg = SerialThread.getInstance().requestConfig();
		if (cfg == null) {
			new ShowMessage("Error: Reading configuration from charger failed");
			return;
		}
		System.out.println("Calibration.readConfig: Read data: "+cfg);
		if (cfg.length() < CALIB_HEXBYTES_RW) {
			System.out.println("Calibration.readConfig: Error: Read data too short");
			new ShowMessage("Error: Read configuration from charger is too short");			
			return;
		}
		//read in and check if valid
		String errorBytes = new String();
		for (int i = 0; i < CALIB_ENTRIES*2; i++) {
			int al = Integer.parseInt(cfg.substring(i*4, i*4+2), 16);
			int ah = Integer.parseInt(cfg.substring(i*4+2, i*4+4), 16);
			int bl = Integer.parseInt(cfg.substring(i*4, i*4+2), 16);
			int bh = Integer.parseInt(cfg.substring(i*4+2, i*4+4), 16);
			//set value
			dataarray[i] = al+(ah<<8);
			//check if valid
			if (((al & 0xff) & (~(bl & 0xff))) != 0) {
				if (errorBytes.length() > 0) {
					errorBytes += ", ";
				}
				errorBytes += i*2;
			}
			if (((ah & 0xff) & (~(bh & 0xff))) != 0) {
				if (errorBytes.length() > 0) {
					errorBytes += ", ";
				}
				errorBytes += i*2+1;
			}
		}
		if (errorBytes.length() > 0) {
			new ShowMessage("Error: Databytes "+errorBytes+" do not fit with the complementary bytes.\n\r This indicates an EEProm corruption in the charger");
		} else
			new ShowMessage("All data read and are valid");
		setGuiValues();	
	}

	private void writeConfig() {
		boolean success = readValuesFromGui();
		if (!success) //reading user input failed
			return;
		//generate String for sending;
		String hexdata = new String();
		for (int i = 0; i < dataarray.length; i++) {
			int ah = (dataarray[i]>>8) & 0xff;
			int al = dataarray[i] & 0xff;
			hexdata +=  String.format("%1$02X%2$02X", al, ah); //%1$x2 = first argument, leading zeros, hexadecimal, 2 digits
		}
		System.out.println("Calibration.writeConfig: Writing bindata: "+hexdata);
		//tell SerialThread
		boolean succ = SerialThread.getInstance().writeConfig(hexdata);
		if (succ) {
			new ShowMessage("Configuration successfully written to charger and will be used after next reset.");
		} else
			new ShowMessage("Error: Writing configuration into charger failed.");
	}
	
	private boolean readValuesFromGui() {
		try {
			convertAndStore(Double.parseDouble(cw.getEdval1Text()), 0);
			convertAndStore(Double.parseDouble(cw.getEdval2Text()), 2);
			convertAndStore(1/Double.parseDouble(cw.getEdval3Text()), 4);
			convertAndStore(1/Double.parseDouble(cw.getEdval4Text()), 6);
			dataarray[8] = (int)(Double.parseDouble(cw.getEdval5Text())*1000);
			dataarray[9] = 1; //currently unused
			applyInverted();
			setGuiLabelValues();
		} catch (NumberFormatException e) {
			e.printStackTrace();
			new ShowMessage("Error: "+e.getMessage() +": Can not be converted into a number");
			return false;
		}
		return true;
	}
	
	private void convertAndStore(double value, int index) {
		//calculate best fitting a/b value
		double error = Double.MAX_VALUE;
		int besta = 1, bestb = 1;
		//TODO there has to be a better way, than a loop
		int b = 1;
		if (value < 0) 
			new ShowMessage("Warning: All values have to be positive. This is not the case.");
		while ((error > 0) && (b <= 0xffff)) {
			int a = Math.max(Math.min((int)((value*((double)b))+0.5), 0xffff), 0); //convert, get denominator, limmit from 0 to 0xffff
			double nerror = Math.abs(((double)a)/((double)b) -value);
			if ((nerror < error) & (a >= 0)) {
				error = nerror;
				besta = a;
				bestb = b;
			}
			if (a >= 0xffff) //all following numbers will be even greater
				break;
			b++;
		}
		dataarray[index] = besta & 0xffff;
		dataarray[index+1] = bestb & 0xffff;
		System.out.println("Calibration.convertAndStore: Using: "+besta+"/"+bestb+" error: "+error);
	}
	
	private void saveConfig() {
		boolean success = readValuesFromGui();
		if (!success) //reading user input failed
			return;
		//determine filename to use
    File file;
    ChooseFile cf = new ChooseFile(".cfg", ChooseFile.SAVE, ConfigFile.getInstance().getSettings("CfgDirName"), null);
    if (cf.getFile() == null) { //user selected abort
    	return;
    }
    file = cf.getFile();
    System.out.println("Calibration.saveConfig: Using filename: "+file);
    ConfigFile.getInstance().setSettings("CfgDirName",cf.getDirectory());
    //open the needed streams
    FileOutputStream srcwriter;
    try {
      srcwriter = new FileOutputStream(file);
    } catch (FileNotFoundException e) { //we want to create the file anyway, so this will not happen
      e.printStackTrace();
      return;
    }
    DataOutputStream os;
    os = new DataOutputStream(new BufferedOutputStream(srcwriter));
    //write to the stream
    try {
      os.writeUTF(FILE_IDENTIFIER);
    	for (int i = 0; i < dataarray.length; i++) {
    		os.writeInt(dataarray[i]);
    	}
    } catch (IOException e) {
    	e.printStackTrace();
    	System.out.println("Calibration.saveConfig: Error: Got an IOException while writing to stream");
    	new ShowMessage("Error: Writing configuration into stream failed");
    }
    //close the streams
    try {
      os.close();
    } catch (IOException e) {
    	System.out.println("Calibration.saveConfig: Error: Got an IOException while closing stream");
    	new ShowMessage("Error: Closing stream for the configuration failed");    	
      e.printStackTrace();
    }
	}
	
	private void openConfig() {
		//determine filename to use
    File file;
    ChooseFile cf = new ChooseFile(".cfg", ChooseFile.OPEN, ConfigFile.getInstance().getSettings("CfgDirName"), null);
    if (cf.getFile() == null) { //user selected abort
    	return;
    }
    file = cf.getFile();
    System.out.println("Calibration.openConfig: Using filename: "+file);
    ConfigFile.getInstance().setSettings("CfgDirName",cf.getDirectory());
    //open the needed streams
    FileInputStream srcreader;
    try {
      srcreader = new FileInputStream(file);
    } catch (FileNotFoundException e) { //should not happen with an user selected filename
      e.printStackTrace();
      return;
    }
    DataInputStream is;
    is = new DataInputStream(new BufferedInputStream(srcreader));
    //write to the stream
    try {
    	String identifier = is.readUTF();
      if (identifier.equals(FILE_IDENTIFIER)) {  
      	for (int i = 0; i < dataarray.length; i++) {
      		dataarray[i] = is.readInt();
      	}
      } else
      	new ShowMessage("Error: Wrong or unsupported file format.");      	
    } catch (IOException e) {
    	e.printStackTrace();
    	System.out.println("Calibration.openConfig: Error: Got an IOException while reading from stream");
    	new ShowMessage("Error: Reading configuration from stream failed");
    }
    //close the streams
    try {
      is.close();
    } catch (IOException e) {
    	//who cares... we have already everything done we could do
    }
    //show to the user
    setGuiValues();
	}
	
}
