package control;

/* Charge Control Version 1.1 (c) 2008 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

import java.awt.image.BufferedImage;
import java.util.Vector;

import javax.swing.ImageIcon;


/**
 * Determines if it is necessary to update the graphic
 * @author Malte Marwedel
 *
 */
public class GraphDrawer {

  boolean hasUpdate = false;
  boolean needsLightUpdate = false;
  boolean needsMediumUpdate = false;
  boolean needsFullUpdate = false;
  
  private Vector<Double> redGraph;
  private Vector<Double> greenGraph;
  private Vector<Double> blueGraph;
  private Vector<Double> redGraphDrawn;
  private Vector<Double> greenGraphDrawn;
  private Vector<Double> blueGraphDrawn;
  
  private int sizex = 1;
  private int sizey = 1;
  
  private int linePrel = -1;
  private int linePrelDrawn = -1;  
  
  private BufferedImage screeni;
  private ImageIcon graphimage;
  
  public GraphDrawer() {
    needsFullUpdate = true;
  }

  /**
   * Tells if the graphic needs to be redrawn since last call to this function.
   * @return
   */
  public boolean isHasUpdated() {
    boolean h = hasUpdate;
    hasUpdate = false;
    return h;
  }
  
  public void updateData(Vector<Double> red, Vector<Double> green, Vector<Double> blue) {
  	redGraph = red;
  	greenGraph = green;
  	blueGraph = blue;
  	if ((graphDiffer(redGraph, redGraphDrawn)) || (graphDiffer(greenGraph, greenGraphDrawn))
  	   || (graphDiffer(blueGraph, blueGraphDrawn))) {
  		needsLightUpdate = true;
  		hasUpdate = true;
  	}
  }
  
  private boolean graphDiffer(Vector<Double> a, Vector<Double> b) {
  	if ((a == null) && (b == null)) {
  		return false;
  	}
  	if ((a == null) && (b != null) || (a != null) && (b == null)) {
  		return true;
  	}
  	if (a.size() != b.size()) {
  		return true;
  	}
  	for (int i = 0; i < a.size(); i++) {
  		if ((a.get(i).equals(b.get(i))) == false) {
  			return true;
  		}
  	}
  	return false;
  }
  
  public void updateSize(int sx, int sy) {
    if ((sx != sizex) || (sy != sizey)) {
     sizex = sx;
     sizey = sy;
     hasUpdate = true;
     needsFullUpdate = true;
    }
  }
  
  public ImageIcon getGraphic() {
    fullUpdate();
    mediumUpdate();
    lightUpdate();
    return graphimage;
  }
  
  private void fullUpdate() {
    if (needsFullUpdate) {
      screeni = new BufferedImage(sizex, sizey, BufferedImage.TYPE_INT_RGB);
      graphimage = new ImageIcon(screeni);
      needsMediumUpdate = true;
      needsFullUpdate = false;
    }
  }

  private void mediumUpdate() {
    if (needsMediumUpdate) {
      for (int x = 0; x < sizex; x++) {
        for (int y = 0; y < sizey; y++) {
          screeni.setRGB(x, y, 0xffffff);
        }
      }
      needsLightUpdate = false;
      drawGraphs();
    }
  }
  
  private void lightUpdate() {
    if (needsLightUpdate) {
    	removeGraphs();
      drawGraphs();
      //System.out.println("Light update");
    }
  }
  
  private void drawGraphs() {
  	if (getDataSize() == 0) { //no data
  		return;
  	}
    double xscaler = ((double)sizex)/((double)getDataSize());
    double yscaler = sizey-1; 
    if (xscaler > 1.0) { //one datapoint stretches multiple pixel
      for (int x = 0; x < sizex; x++) {
        if (redGraph != null) {
        	Double d = redGraph.get((int)((double)x/xscaler));
        	drawPixel(x, d*yscaler, 0xff0000, true);
        }
        if (greenGraph != null) {
        	Double d = greenGraph.get((int)((double)x/xscaler));
        	drawPixel(x, d*yscaler, 0x00ff00, true);
        }
        if (blueGraph != null) {
        	Double d = blueGraph.get((int)((double)x/xscaler));
        	drawPixel(x, d*yscaler, 0x0000ff, true);
        }
      }
    } else { //multiple datapoints on one pixel. -> go to every datapoint
      for (int i = 0; i < getDataSize(); i++) {
        int x = (int)(((double)i)*xscaler+0.5);
        if (redGraph != null) drawPixel(x, ((double)redGraph.get(i))*yscaler, 0xff0000, true);
        if (greenGraph != null) drawPixel(x, ((double)greenGraph.get(i))*yscaler, 0x00ff00, true);
        if (blueGraph != null) drawPixel(x, ((double)blueGraph.get(i))*yscaler, 0x0000ff, true);
      }   
    }
    //draw line
    if (linePrel >= 0) {
    	int linePx = (int)(((double)linePrel)*xscaler);
    	if (linePx < sizex) {
    		for (int i = 0; i < sizey; i++) {
    			screeni.setRGB(linePx, i, 0x000000);
    		}
    	}
    }
    redGraphDrawn = redGraph;
    greenGraphDrawn = greenGraph;
    blueGraphDrawn = blueGraph;
    linePrelDrawn = linePrel;
    needsLightUpdate = false;
    needsMediumUpdate = false;
    needsFullUpdate = false;
  }

  private void removeGraphs() {
  	if (getDrawnDataSize() == 0) { //no data
  		return;
  	}
    double xscaler = ((double)sizex)/((double)getDrawnDataSize());
    double yscaler = sizey-1; 
    if (xscaler > 1.0) { //one datapoint stretches multiple pixel
      for (int x = 0; x < sizex; x++) {
        if (redGraphDrawn != null) {
        	Double d = redGraphDrawn.get((int)((double)x/xscaler));
        	drawPixel(x, d*yscaler, 0xffffff, false);
        }
        if (greenGraphDrawn != null) {
        	Double d = greenGraphDrawn.get((int)((double)x/xscaler));
        	drawPixel(x, d*yscaler, 0xffffff,false);
        }
        if (blueGraphDrawn != null) {
        	Double d = blueGraphDrawn.get((int)((double)x/xscaler));
        	drawPixel(x, d*yscaler, 0xffffff, false);
        }
      }
    } else { //multiple datapoints on one pixel. -> go to every datapoint
      for (int i = 0; i < getDrawnDataSize(); i++) {
        int x = (int)(((double)i)*xscaler+0.5);
        if (redGraphDrawn != null) drawPixel(x, ((double)redGraphDrawn.get(i))*yscaler, 0xffffff, false);
        if (greenGraphDrawn != null) drawPixel(x, ((double)greenGraphDrawn.get(i))*yscaler, 0xffffff, false);
        if (blueGraphDrawn != null) drawPixel(x, ((double)blueGraphDrawn.get(i))*yscaler, 0xffffff, false);
      }   
    }
    //remove line
    if (linePrelDrawn >= 0) {
    	int linePx = (int)(((double)linePrelDrawn)*xscaler);
    	if (linePx < sizex) {
    		for (int i = 0; i < sizey; i++) {
    			screeni.setRGB(linePx, i, 0xffffff);
    		}
    	}
    }
    redGraphDrawn = null;
    greenGraphDrawn = null;
    blueGraphDrawn = null;
  }
  
  private void drawPixel(int x, Double y, int color, boolean addcolor) {
    int newcolor = 0x000000;
    int iy = (int)(y+0.5);
    if ((iy >= 0) && (iy < sizey) && (x >= 0) && (x < sizex)) {
    	int oldcolor = screeni.getRGB(x, sizey-iy-1) & 0xffffff; //get old value and remove alpha channel
    	if ((addcolor) && (oldcolor != 0xffffff)) {
          /*make not an additive but a special sort of
           * subtractive rgb color calculations, because otherwise: r+b+g= white = background color = invisible graph
           * 
           * This function makes some assumptions about the used colors:
           * only one r or b or g is set at one time. they are always set with 0xff and the background is 0xffffff = white
           * if two of the base colors are set, this results in 0x7f for each color. if three are set, this results in 0x01 for each color
           * (this may not be 0x00 because then we could not decide if it was set next round
           * 
           * You may call this function either genius or stupid ;-)
           * */
          //check which colors were bright and are not the now adding color
    	  int rold = (oldcolor & ~color) & 0xff0000;
          int gold = (oldcolor & ~color) & 0x00ff00;
          int bold = (oldcolor & ~color) & 0x0000ff;
          //count how many they are
          int colorsBrightBefore = 0; 
          if (rold != 0x0) colorsBrightBefore++;
          if (gold != 0x0) colorsBrightBefore++;
          if (bold != 0x0) colorsBrightBefore++;
          if (colorsBrightBefore == 0) {
            newcolor = color;
          } else if (colorsBrightBefore == 1) {
            newcolor = 0x7f7f7f & (oldcolor | color); //merge colors and set MSB of each color to zero
          } else  if (colorsBrightBefore >= 2) {
            newcolor = 0x010101;
          }
        } else
          newcolor = color;
    	//System.out.println("Set color on "+x+","+iy+" from "+Integer.toHexString(oldcolor)+" to "+Integer.toHexString(newcolor));
      screeni.setRGB(x, sizey-iy-1, newcolor);
    }
  }
  
  private int getDataSize() {
  	if (redGraph != null) {
  		return redGraph.size();
  	}
  	if (greenGraph != null) {
  		return greenGraph.size();
  	}
  	if (blueGraph != null) {
  		return blueGraph.size();
  	}
  	return 0;
  }

  private int getDrawnDataSize() {
  	if (redGraphDrawn != null) {
  		return redGraphDrawn.size();
  	}
  	if (greenGraphDrawn != null) {
  		return greenGraphDrawn.size();
  	}
  	if (blueGraphDrawn != null) {
  		return blueGraphDrawn.size();
  	}
  	return 0;
  }

	public void setLine(int relPos) {
		linePrel = relPos;
		if (relPos != linePrelDrawn) {
			needsLightUpdate = true;
    	hasUpdate = true;
		}
	}
  
}
