package control;

/* Charge Control Version 1.1 (c) 2008 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

import gnu.io.CommPort;

import java.io.IOException;
import java.util.Vector;

/**
 * This is a thread for communicating with the serial cable.
 * It provides a status value for the charger and all incoming bytes are placed into a line based FIFO
 * Moreover commands for reading and writing the configuration are provided
 * 
 * This class implements the singleton pattern
 * 
 * @author Malte Marwedel
 *
 */

public class SerialThread implements Runnable {
	
	private static SerialThread st;
	
	private final int MWIURS = 3000; //Milis without input until reset state
	private final int POLL_FREQ_MS = 20;
	
	public final static int CHARGER_UNKNOWN = 0;
	public final static int CHARGER_IDLE = 1;
	public final static int CHARGER_PROBE = 2;
	public final static int CHARGER_CHARGING = 3;
	public final static int CHARGER_DISCHARGING = 4;
	
	private volatile int chargerstate = CHARGER_UNKNOWN;
	private volatile Vector<String> fifo;
	

	private volatile boolean shouldRun = true;
	private volatile boolean iamRunning = false;
	private int noactionMilis = 0;
	
	private Thread sthr;
	
	private volatile CommPort commport;
	
	//some variables to tell the thread what he should do
	private volatile boolean wishReset = false;
	
	private volatile boolean getConfig = false;
	private volatile String readedConfig;
	
	private volatile String toWriteConfig;
	private volatile boolean writeConfigDone = false;
	
	
	
	public static SerialThread getInstance() {
		if (st == null) {
			st = new SerialThread();
		}
		return st;
	}
	
	private SerialThread() {
		fifo = new Vector<String>();
	}
	
	public void run() {
		iamRunning = true;
		String accumu = new String();
		while(shouldRun) {
			int avar;
			try {
				avar = commport.getInputStream().available();
				//System.out.println("Available on input "+avar);
				if (avar > 0) {
					byte arr[] = new byte[avar];
					commport.getInputStream().read(arr, 0, avar);
					String buff = new String(arr);
					buff = buff.replaceAll("\r", "");
					//System.out.println("Raw: '"+temp+"'"+"newline at: "+temp.indexOf('\n'));
					int k = 0;
					int l;
					while ((l = buff.indexOf('\n', k)) >= 0) { //for every newline do
						accumu += buff.substring(k, l);
						//System.out.println("add to fifo: '"+accumu+"'");
						updateStatus(accumu);
						putToFifo(accumu);
						accumu = new String();
						k = l+1;
					}
					if (k != buff.length()) { //if read does not end with a \n 
						accumu += buff.substring(k, buff.length()); //safe the rest for next round
					}
				}
				sendResetIfWished();
				getConfigIfWished();
				writeConfigIfWished();
			} catch (IOException e) {
				avar = 0;
				System.out.println("Serial.Thread.run: Error: "+e.getMessage());
			}
			try {
	      Thread.sleep(POLL_FREQ_MS);
      } catch (InterruptedException e) {
	      //who cares
      }
      //reset state if no info came in
      if (noactionMilis < MWIURS) {
      	noactionMilis += POLL_FREQ_MS;
			} else {
				chargerstate = CHARGER_UNKNOWN;
			}
		}
		if (accumu.length() > 0) {
			putToFifo(accumu); //add possible unfinished strings
		}
		chargerstate = CHARGER_UNKNOWN;
		iamRunning = false;
	}

	private void writeConfigIfWished() {
		if (toWriteConfig != null) {
			String wr = toWriteConfig;
			try {
				commport.getOutputStream().write("Pd".getBytes()); //Pd= put data
				try {
		      Thread.sleep(10); //give the charger some time
	      } catch (InterruptedException e) {
		      //who cares
	      }				
				for (int i = 0; i < wr.length()/2; i++) {
					commport.getOutputStream().write(wr.substring(i*2, i*2+2).getBytes());
					//wait for acknowledgment
					int j = 0;
					while (commport.getInputStream().available() == 0) {
						j++;
						 //wait up to 500ms for an ack
						try {
				      Thread.sleep(10);
			      } catch (InterruptedException e) {
				      //who cares
			      }
			      if (j >= 50) { //too long wait, abort this function
							toWriteConfig = null; //do not write the config again
			      	System.out.println("SerialThread.writeConfigIfWished: Error, no data received from charger");
			      	return;
			      }
			    } 
					if (commport.getInputStream().read() != 'A') {
						toWriteConfig = null; //do not write the config again
						System.out.println("SerialThread.writeConfigIfWished: Error, invalid acknowledgment received from charger");
						return;
					}
					//System.out.println("SerialThread.writeConfigIfWished: Agreed on hexbyte "+i);
				}
				toWriteConfig = null; //do not write the config again
				writeConfigDone = true;
				System.out.println("SerialThread.writeConfigIfWished: All bytes written");
				//clear input stream
				try {
		      Thread.sleep(30); //give the charger some time
	      } catch (InterruptedException e) {
		      //who cares
	      }				
				int avar = commport.getInputStream().available(); //i expect an '\n\r', but not more 
				for (int i = 0; i < avar; i++) {
					commport.getInputStream().read();
				}
				if (avar > 2)
					System.out.println("SerialThread.writeConfigIfWished: Warning: Got "+(avar-2)+" unused bytes from charger");
			} catch (IOException e) {
				System.out.println("SerialThread.writeConfigIfWished: Error while writing to output stream, got an IOException");
			}
		}
	}

	private void getConfigIfWished() {
		if (getConfig) {
			try {
				commport.getOutputStream().write("G".getBytes());
			} catch (IOException e) {
				System.out.println("SerialThread.getConfigIfWished: Error: Writing get to output stream failed");
			}
			for (int i = 0; i < 30; i++) { //3 seconds time
				try {
		      Thread.sleep(100);
	      } catch (InterruptedException e) {
		      //who cares
	      }
	      int avar;
				try {
					avar = commport.getInputStream().available();
				} catch (IOException e) {
					System.out.println("SerialThread.getConfigIfWished: Error: Accessing input stream failed");
					return;
				}
				if (avar >= (Calibration.CALIB_HEXBYTES_RW)) { //if enough data are read
					//got available bytes
					byte[] readed = new byte[avar];
					try {
						commport.getInputStream().read(readed, 0, avar);
					} catch (IOException e) {
						System.out.println("SerialThread.getConfigIfWished: Error: Reading from input stream failed");
						return;
					}
					readedConfig = new String(readed);
					getConfig = false;					
				}
			}
		}
	}

	private void sendResetIfWished() {
		if (wishReset) {
			wishReset = false;
			try {
				commport.getOutputStream().write("R".getBytes());
			} catch (IOException e) {
				System.out.println("SerialThread.sendResetIfWished: Error: Writing reset to output stream failed");
			}
		}
	}
	
	public void updateStatus(String str) {
		/* Examples:
		 * I:3;00159;0000
		 * I:2;12718;0003
		 */
		if ((str.length() == 14) && (str.charAt(0) == 'I') &&  (str.charAt(1) == ':'))  {
			if (str.charAt(2) == '1') {
				chargerstate = CHARGER_PROBE;
				noactionMilis = 0;
			}
			if (str.charAt(2) == '2') {
				chargerstate = CHARGER_CHARGING;
				noactionMilis = 0;
			}
			if (str.charAt(2) == '3') {
				chargerstate = CHARGER_DISCHARGING;
				noactionMilis = 0;
			}
		}
		/*
		 * Examples:
		 * T:A;00;00;00000;00
		 */
		if ((str.length() == 18) && (str.charAt(0) == 'T') &&  (str.charAt(1) == ':'))  {
			chargerstate = CHARGER_IDLE;
			noactionMilis = 0;
		}
	}
	
	public void start(CommPort cp) {
		commport = cp;
		shouldRun = true;
		sthr = new Thread(this);
		sthr.start();
		while(!iamRunning) {			
			try {
	      Thread.sleep(1);
      } catch (InterruptedException e) {
	      //who cares
      }
		}
	}
	
	public void stop() {
		shouldRun = false;
		while(iamRunning) {
			try {
	      Thread.sleep(1);
      } catch (InterruptedException e) {
	      //who cares
      }
		}
	}
	
	public String getChargerstateDescription() {
		switch (chargerstate) {
			case CHARGER_IDLE: return "idle";
			case CHARGER_PROBE: return "probe";
			case CHARGER_CHARGING: return "charging";
			case CHARGER_DISCHARGING: return "discharging";
		}
		return "unknown";
	}

	public int getChargerstate() {
		return chargerstate;
	}
	
	public synchronized int getFifoSize() {
		return fifo.size();
	}
	
	private synchronized void putToFifo(String str) {
		fifo.add(str);
	}
	
	public synchronized String getFromFifo() {
		if (fifo.size() > 0) {
			String str = fifo.get(0);
			fifo.remove(0);
			return str;
		}
		System.out.println("SerialThread.getFromFifo: Warning: Tried to read from FIFO, but FIFO is empty");
		return null;
	}

	public void sendReset() {
		wishReset = true;			
	}
	
	/**
	 * Gets the config which the charger is using. It may take up to 5 seconds until this function returns
	 * @return The read config, null if reading failed
	 */
	public String requestConfig() {
		if (shouldRun && iamRunning) {
			readedConfig = null;
			getConfig = true; //will be set to false as soon as this is done
			for (int i = 0; i < 50; i++) { //5 seconds time
				try {
		      Thread.sleep(100);
	      } catch (InterruptedException e) {
		      //who cares
	      }
				if (getConfig == false) { //config read
					String tmp = readedConfig;
					readedConfig = null;
					return tmp;
				}
			}
		}	
		getConfig = false;
		return null; //reading failed
	}
	
	/**
	 * Writes a new config to the EEProm, it may take up to 5 seconds until this function returns
	 * @param newConfig
	 * @return true on success, false if failed (like no response, or closed port...) 
	 */
	public boolean writeConfig(String newConfig) {
		writeConfigDone = false;
		if (shouldRun && iamRunning) {
			toWriteConfig = newConfig;
			for (int i = 0; i < 50; i++) { //5 seconds time
				try {
		      Thread.sleep(100);
	      } catch (InterruptedException e) {
		      //who cares
	      }
				if (writeConfigDone == true) { //config read
					writeConfigDone = false;
					toWriteConfig = null;
					return true;
				}
			}
		}
		toWriteConfig = null;
		return false;
	}
}
