/* ad_converter.c
Everything which has to do with A/D conversion

Copyright (C) 2006-2007 by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  US
*/

#include "main.h"

/*The hires values consists of the average of 32 normal measurements
  the disadvantage is the slower update */

u16 volatile m_uppervoltage;		//measured upper output voltage
u16 volatile m_hiresuppervoltage;	//measured hires upper output voltage
u16 volatile m_lowervoltage;		//measured lower output voltage
u16 volatile m_hireslowervoltage;	//measured hires lower output voltage
u16 volatile m_vccvoltage;		//measured Vcc voltage


u08 volatile updated;
u08 volatile cycle; //odd: lowersum, even:uppersum
u16 volatile lowersum_cnt;
u16 volatile uppersum_cnt;
u16 volatile lowersum_raw;
u16 volatile uppersum_raw;
u16 volatile bandgap_raw;


//the common threading is too slow or the amount of conversions
ISR(ADC_vect) {
u08 cycle_l = cycle;		//local variable improves speed
if (cycle_l <= 128) {
  if (cycle_l & 1) {		//odd -> lower sum has been converted
    lowersum_cnt += ADCW;
    ADMUX = (1<< REFS0) | UPPER_PIN; //upper voltage comes next
  } else {			//even -> upper sum has been converted
    uppersum_cnt += ADCW;
    ADMUX = (1<< REFS0) | LOWER_PIN; //lower voltage comes next
  }
  if (cycle_l == 128) {
    ADMUX = (1<< REFS0) | BANDGAP_PIN; //measure internal vcc
  }
} else if (cycle >= 136) {
  /*The bandgap has been measured 8 times now because the input seems to have
    a very high resistance and adapts very slowly to the right value */
  cycle_l = 0;
  lowersum_raw = lowersum_cnt;
  lowersum_cnt = 0;
  uppersum_raw = uppersum_cnt;
  uppersum_cnt = 0;
  bandgap_raw = ADCW;
  ADMUX = (1<< REFS0) | LOWER_PIN;
  updated = 1;
}
ADCSRA |= (1<<ADSC);		//start conversion
cycle_l++;			//number of started conversion
cycle = cycle_l;		//write back local variable
}

void ad_thread(void) {
u32 hiresuppersum = 0, hireslowersum = 0;
u08 hiresmeasures = 0;
//A/D converter with a prescaler of 64
//resulting in a frequency of 125KHZ (at F_CPU = 8MHZ)
ADCSRA = (1 << ADEN) | (1<< ADIE) | (0x06); //enable with prescaler
ADMUX = (1<< REFS0) | LOWER_PIN;	//set to lower pin
cycle = 1;				//first conversion will be started
ADCSRA |= (1<<ADSC);			//start first conversion
for (;;) {
  do {			//wait until updated
    sched();
  } while (updated == 0);
  //to prevent an ADC interrupt from updating while reading the variables
  cli();
  u16 lowersum = lowersum_raw;
  u16 uppersum = uppersum_raw;
  u16 bandgap = bandgap_raw;
  updated = 0;
  sei();
  //rs232_print_number(bandgap); //increase stack before enabling
  //calculate the upper voltage in mV
  /*To do this the uppersum value has to be
    divided trough the maximum A/D value 1023
    multiplied with the reference voltage 5.0V
    divided with the R7=10k resistor
    multiplied by the sum of the R4=30k and R7=10k resistors
    multiplied by 1000 to convert V to mV
    divided the the amount of samples MEASURE_SUM
    uppersum/1023.0*5.0/10.0*(10.0+30.0)*1000.0/((float)MEASURE_SUM);
    since this is a lot of calculation we compact this to:
    uppersum*200000/654720 = uppersum*1250/4092
  */
  //uppersum = multiplyanddivide(uppersum,1250,4092);
  uppersum = eep_multiplyfactor(uppersum, CALIB_UPPERSUM);
  m_uppervoltage = uppersum;
  //calculate the lower voltage in mV
  /* the lower sum has to be calculated a similar way, however there are no
    resistors for voltage division on front of the A/D converter.
    lowersum
    divided trough the maximum A/D value 1023
    multiplied with the reference voltage 5.0V
    multiplied by 1000 to convert V to mV
    divided the the amount of samples MEASURE_SUM
    lowersum/1023.0*5.0*1000.0/((float)MEASURE_SUM));
    since this is a lot of calculation we compact this to:
    lowersum*5000/654720 = lowersum*625/8184
  */
  //lowersum = multiplyanddivide(lowersum, 625, 8184);
  lowersum = eep_multiplyfactor(lowersum, CALIB_LOWERSUM);
  m_lowervoltage = lowersum;
  //update hires voltages
  hiresuppersum += uppersum;
  hireslowersum += lowersum;
  hiresmeasures++;
  if (hiresmeasures == 32) {
    hiresmeasures = 0;
    m_hiresuppervoltage = hiresuppersum/32;
    hiresuppersum = 0;
    m_hireslowervoltage = hireslowersum/32;
    hireslowersum = 0;
  }
  /*Measuring the Vcc works by measuring the known internal reference voltage.
    By this way you can calculate back to what A/D reference voltage (which is
    Vcc) voltage we have.
    The reference should be 1,1V (1.092V) and can vary according to the
    datasheet +- 0.1V.
    Vcc = 1,092*1023/ADCW => Vcc = 1117.116/ADCW
    Because the values are in mV: 1117116/ADCW
 */
  m_vccvoltage = 1117116/bandgap;
}
}

u16 get_current(void) {
/* returns the charging or discharging current
in modes where no current should flow or the current cant be estimated,
0 is returned. So this function may not produce always a correct output */
u08 op_mode_l = op_mode;
u16 res = 0;
u08 calibval = CALIB_INVALID;
//On charging the current flows through R2
if (op_mode_l == OP_CHARGE)
  calibval = CALIB_R2;
//On discharging the current flows through R3
if (op_mode_l == OP_DISCHARGE)
  calibval = CALIB_R3;
if (calibval != CALIB_INVALID)
  res = eep_multiplyfactor(m_lowervoltage, calibval); //I=U/R=U*(R^-1)
return res;
}

u16 get_outputvoltage(void) {
/*returns the output voltage - as long as the voltage can be determined
*/
u08 op_mode_l = op_mode;
u16 res = 0;
if ((op_mode_l == OP_DISCHARGE) || (op_mode_l == OP_CHARGE)) {
  res = m_uppervoltage;
}
if (op_mode_l == OP_CHARGE) {
  res -= min(m_lowervoltage, res); //the min prevents underflows
}
return res;
}

u16 get_hiresoutputvoltage(void) {
/*returns the output voltage - as long as the voltage can be determined
*/
u08 op_mode_l = op_mode;
u16 res = 0;
if ((op_mode_l == OP_DISCHARGE) || (op_mode_l == OP_CHARGE)) {
  res = m_hiresuppervoltage;
}
if (op_mode_l == OP_CHARGE) {
  res -= min(m_hireslowervoltage, res); //the min prevents underflows
}
return res;
}

u16 get_uppervoltage(void) {
return m_uppervoltage;
}

u16 get_lowervoltage(void) {
return m_lowervoltage;
}


u16 get_hiresuppervoltage(void) {
return m_hiresuppervoltage;
}

u16 get_hireslowervoltage(void) {
return m_hireslowervoltage;
}


u16 get_vccvoltage(void) {
return m_vccvoltage;
}
