/* chargedischarge.c
The main routines for charging and discharging.

Copyright (C) 2007-2008 by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  US
*/

#include "main.h"

u16 c_limitheat(u16 current, u16 voltage) {
//returns the lower number of current, and MAX_HEAT_POWER divided by the voltage
//the return value represents the maximum current for charging or discharging
return min(multiplyanddivide(MAX_HEAT_POWER, 1000, voltage), current);
}

static void c_chargedischarge_endwait(u08 ret_code) {
/*waits for a key press if this is the last process before going back to the
main menu */
waitms_sched(750);
if (ret_code == 0) {
  //Let the user show the statistics because it is going back to the main menu
  key_wait_until_pressed();
}
}

//a global variable saves memory.
struct daytime thetime;
u32 timing; //for running some parts once every second

//an other one for saving memory
u32 current_sum; //sum of all charged or discharged mA over the time

//two status variables for c_display()
u08 show_current = 0;
u32 display_update;

static void c_reset_counters(void) {
thetime.seconds = 0;
thetime.minutes = 0;
thetime.hours = 0;
timer_setup(&timing, 1000);	//one second
timer_setup(&display_update, 20);	// update display soon
show_current = 0; //not a counter, but a status
current_sum = 0;
}

//common for charging and discharging, should be called once per second
static u16 c_update_mAh(void) {
//returns the value of the currently charged/discharged mAh
current_sum += get_current();
return (current_sum / 3600);
}

static u08 c_update_time(void) {
//returns 1: if updated, 0: of not updated
u08 res = 0;
if (timer_renew(&timing, 1000)) { //if one second is over -> update
  thetime.seconds++;
  u08 minutes = thetime.minutes;
  u08 hours = thetime.hours;
  if (thetime.seconds >= 60) {
    thetime.seconds = 0;
    minutes++;
    if (minutes >= 60) {
      minutes = 0;
      hours++;
    }
  }
  thetime.minutes = minutes;
  thetime.hours = hours;
  res = 1;
}
return res;
}

static u08 c_sendbuf[20];

//Sends the values of the beginning charging/discharging process by RS232
static void c_print_startinfo(u08 function) {
//format: B:cc;SSSSS;CCCCC;PP
c_sendbuf[0] = 'B';
c_sendbuf[1] = ':';
c_sendbuf[4] = ';';
c_sendbuf[10] = ';';
c_sendbuf[16] = ';';
c_sendbuf[19] = '\0';
wordtostr(c_sendbuf, battery_get_cells(), 2, 2); //the 'cc'
//the 'SSSSS' speed
if (function == OP_CHARGE) {
  wordtostr(c_sendbuf, battery_get_chargingcurrent(), 5, 5);
} else if (function == OP_DISCHARGE) {
  wordtostr(c_sendbuf, battery_get_dischargingcurrent(), 5, 5);
} else
  wordtostr(c_sendbuf, 65500, 5, 5); //this may not happen
wordtostr(c_sendbuf, battery_get_capacity(), 5, 11); //the 'CCCCC'
wordtostr(c_sendbuf, battery_get_chargedpercent(), 2, 17); //the 'PP'
rs232_println(c_sendbuf);
}

//Sends the values at the end of charging/discharging process by RS232
static void c_print_stopinfo(u08 function, u08 abort_cond) {
//format: T:R;MM;HH;CCCCC;PP
c_sendbuf[0] = 'T';
c_sendbuf[1] = ':';
c_sendbuf[2] = abort_cond; // the 'R' reason for aborting
c_sendbuf[3] = ';';
c_sendbuf[6] = ';';
c_sendbuf[9] = ';';
c_sendbuf[15] = ';';
c_sendbuf[19] = '\0';
wordtostr(c_sendbuf, thetime.minutes, 2, 4); //the 'MM'
wordtostr(c_sendbuf, thetime.hours, 2, 7); //the 'HH'
//the 'CCCCC' charged/discharged capacity
if (function == OP_CHARGE) {
  wordtostr(c_sendbuf, battery_get_charged(), 5, 10);
} else if (function == OP_DISCHARGE) {
  wordtostr(c_sendbuf, battery_get_discharged(), 5, 10);
} else
  wordtostr(c_sendbuf, 65500, 5, 10); //this may not happen
wordtostr(c_sendbuf, battery_get_chargedpercent(), 3, 16); //the 'PP'
rs232_println(c_sendbuf);
}


const char ch_milliamp_hours[] PROGMEM = "$5mAh  ";
const char ch_milliamps[] PROGMEM = "$5mA   ";
const char ch_percent[] PROGMEM = "$3%";
const char ch_voltage[] PROGMEM = "\nU:$5mV ";
const char ch_time[] PROGMEM = {0x80+26,'$','2',0x80+29,'$','2','h','\0'};

/*A general function for displaying everything while charging/discharging
Menu: B>00000mAh
      U:00000mV xx:xxh
*/
static void c_display(u08 function, u16 charged, u08 percent) {
/*function: is either OP_CHARGE or OP_DISCHAGE and nothing else */
/*running: 1: still charging or discharging
 (OP_PROBE is not used on charging, discharging) */
u08 running = 0;
if ((op_mode == OP_CHARGE) || (op_mode == OP_DISCHARGE))
  running = 1;
//check for keys
if (key_up_pressed()) {
  show_current = 0;
}
if (key_down_pressed()) {
  show_current = 1;
}
//prevent flicker, update only every 200ms while running
if (timer_renew(&display_update, 200) || (running == 0)) {
  //go to display beginning
  lcd_putchar('\r');
  lcd_putchar(BATTERY_SYMB_CHAR); //battery symbol
  //print arrow
  if (running) {
    if (op_mode == OP_CHARGE) {
      lcd_putchar(0x7f);
    } else if (op_mode == OP_DISCHARGE) {
      lcd_putchar(0x7e);
    }
  } else {
    lcd_putchar(' ');
    show_current = 0; //if output is off, no one is interested in reading 0mA
  }
  //print current
  if (show_current) {
    lcd_printf_p(ch_milliamps, get_current(), 0);
  } else
    lcd_printf_p(ch_milliamp_hours, charged, 0);
  //print progress if charging mode
  if (function == OP_CHARGE) {
    lcd_printf_p(ch_percent, percent, 0);
  }
  /*Print voltage only if running, because otherwise it will be read 0mV,
    so the last measured voltage which is printed will still be visible on the
    LCD if charging/discharging has finished */
  if (running)
    lcd_printf_p(ch_voltage, get_hiresoutputvoltage(), 0);
  //print time
  lcd_printf_p(ch_time, thetime.hours, thetime.minutes);
  //let points between the minutes and hours flash
  lcd_putchar(0x9c);
  if ((thetime.seconds % 2) && (running)) { //not display ':' on odd seconds
    lcd_putchar(' ');
  } else //display
    lcd_putchar(':');
}
}


/*Returns when the cell is charged or the user aborted
Menu: B>00000mAh  xxx%
      U:00000mV xx:xxh
or:   B>00000mA   xxx%
      U:00000mV xx:xxh
The charging is terminated by:
The user:       Pressing the left key
Maximum charged mAh: Since the current is constant,
                this is similar to a timeout
Low charging:   If the measured current is below the half of the expected one
-deltaV:        If the voltage is more than CAL_DELTA_VOLT below the voltage was
                once as maximum.

Maximum voltage has be removed as reason for termination because batteries with
a different Ri have very different values here. Second the voltage is not
increasing if the battery is full, so in most of the cases either a fixed
maximum voltage would abort the charge before the battery is full, or would not
abort at all.

Warning: The c_chargenow() function is not made for a very high charging speed.
For the -deltaV detection the voltage is sampled every minute and is compared
every 10% of the charge. So the detection would not work well if there is
10% or more of the battery charged within two minutes. Resulting in a minimum
charging time of 20 minutes for a full charge. However such fast charged can
not be selected anyway.
*/

static u08 c_chargenow_testforstop(u08 percent, u08 swingedin) {
/*Returns 0 if no stop is needed, otherwise a ASCII character is returned
  indicating which stop condition occurred */
u08 abort_cond = 0;
//do -deltaV if active
if (battery_get_deltavuse()) {
  //overwrites abort_cond, so it has to be the first check
  abort_cond = delta_ckeck(percent);
}
if (key_left_pressed()) {	//abort key pressed by user
  abort_cond = 'A';		//Stop condition: Abort
}
if (swingedin) {		//detect too low charge current -> error
  if (battery_get_chargingcurrent() / 2 > get_current()) {
    abort_cond = 'C';		//Stop condition: Current
  }
}
//detect maximum charge amount
if (percent >= 100) {		//if the battery is full
  abort_cond = 'P';		//Stop condition: 100% Percent
}

return abort_cond;
}

u08 c_chargenow(struct menu_selector * the_menu) {
u08 percent =  battery_get_chargedpercent(); //how full the battery is
u16 current_v;			//current voltage of the battery
u16 charged = 0;		//as charged_sum but in mAh
u16 charged_max;		//with overchargefactor, compare with charged
u08 ret_code = the_menu->next;
u08 output_swingedin = 0;	//will be set to 1 after 60 seconds
u08 abort_cond = 0;
lcd_clear();
//calc max charged
charged_max = multiplyanddivide(battery_get_capacity(),
                   battery_get_overchargefactor(), 60);
//setup voltage
current_v = get_uppervoltage();
//limit maximum heat generation
battery_set_chargingcurrent(c_limitheat(battery_get_chargingcurrent(),
                            (maxout_voltage+CAL_VCC_DROP)-current_v));
//print out statistics on RS232
c_print_startinfo(OP_CHARGE);
//start charge
da_charge(battery_get_chargingcurrent());
//setup timer
c_reset_counters();
delta_init();
while (abort_cond == 0) {
  current_v = get_hiresoutputvoltage();		//measure voltage
  abort_cond = c_chargenow_testforstop(percent, output_swingedin);
  //update clock
  if (c_update_time()) {	//this part will be called every second
    charged = c_update_mAh();		//add current mA and get charged mAh
    battery_set_charged(charged);	//for the stats
    //calc the percent's
    percent = battery_get_chargedpercent() +
                                   multiplyanddivide(charged, 100, charged_max);
    if (thetime.seconds == 10) //give 10 seconds for voltage swing in
      output_swingedin = 1; //enable low current stop
  }
  //show on lcd
  c_display(OP_CHARGE, charged, percent);
  sched();
}
da_main();
battery_set_chargedpercent(percent);
if (abort_cond == 'A')
  ret_code = the_menu->prev;
c_print_stopinfo(OP_CHARGE, abort_cond);
c_display(OP_CHARGE, charged, percent);
c_chargedischarge_endwait(ret_code);
return ret_code;
}

/*Returns when the cell is discharged or the user aborted
Start discharge: with 1/2 C current
Stop discharge: at 1/12 C current
*/
u08 c_dischargenow(struct menu_selector * the_menu) {
u16 current_v, stop_v, increase_v;
u16 discharged = 0;
u16 dis_min_current, dis_max_current; 	//the discharging current limits
u08 ret_code = the_menu->next;
u08 output_swingedin = 0;	//will be set to 1 after 10 seconds
u08 abort_cond = 'E';
lcd_clear();
//setup voltage
current_v = get_uppervoltage();
stop_v = CAL_CELL_VERYEMPTY_V * battery_get_cells();
increase_v = CAL_CELL_LOAD_V * battery_get_cells();
//calculate starting discharge value
battery_set_dischargingcurrent(battery_get_capacity()/ 8);
//current where the discharging is stopped
dis_min_current = battery_get_dischargingcurrent() / 2 +1;
//maximum current for discharging
dis_max_current = min (battery_get_capacity()/ 2,
                       CAL_CHRG_SELECT_MAXDSMILLIAMPS-CAL_CHARGE_OFFSET);
//print out stats on RS232
c_print_startinfo(OP_DISCHARGE);
//setup timer
c_reset_counters();
/*loop as long as the discharging current is greater than the minimum stopping
 value
*/
while (dis_min_current < battery_get_dischargingcurrent()) {
  //limit discharging value (by the maximum value set before and the heat
  // generated)
  battery_set_dischargingcurrent(min(dis_max_current,
              c_limitheat(battery_get_dischargingcurrent(),
                      max(battery_get_cells()*CAL_CELL_CHARGED_V, current_v))));
  //set value for new discharging current
  //rs232_print_number(battery_get_dischargingcurrent());
  da_discharge(battery_get_dischargingcurrent());
  if (key_left_pressed()) {	//abort
    ret_code = the_menu->prev;
    abort_cond = 'A';
    break;
  }
  //measure voltage
  current_v = get_outputvoltage();
  //update clock
  if (c_update_time()) {
    discharged = c_update_mAh(); //add current mA and get discharged mAh
    //update current
    if (current_v < stop_v) {
      //reduce current by 20 percent, or at least by 1 mA
      battery_set_dischargingcurrent(min(
               multiplyanddivide(battery_get_dischargingcurrent(), 4, 5),
               battery_get_dischargingcurrent()-1));
    }
    if (current_v > increase_v) {
      //increase current by 20 percent or at least by 1 mA
      battery_set_dischargingcurrent(max(
               multiplyanddivide(battery_get_dischargingcurrent(), 5, 4),
               battery_get_dischargingcurrent()+1));
    }
    if (thetime.seconds == 10) //give 10 seconds for voltage swing in
      output_swingedin = 1;
  }
  if (output_swingedin) {
    //abort if the discharging current is below one mA
    if (get_current() == 0)
      break;
  }
  //print on lcd
  c_display(OP_DISCHARGE, discharged, 0);
  sched();
}
da_main();
battery_set_discharged(discharged);	//for the statistics
battery_set_chargedpercent(0);	//expect the cell to be empty now
c_print_stopinfo(OP_DISCHARGE, abort_cond);
c_display(OP_DISCHARGE, discharged, 0);
c_chargedischarge_endwait(ret_code);
return ret_code;
}
