/* lcd.c

Copyright (C) 2006-2007 by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "main.h"

#define LCD_SIZE 32
u08 volatile lcd_virtual[LCD_SIZE];
u08 volatile lcd_modify_flags[LCD_SIZE/8+1];
u08 lcd_pointer = 0;

static void lcd_set_4bit(u08 ch) {
PORTB &= ~(1<<PB5);	//clear data pins
PORTB &= ~(1<<PB4);
PORTB &= ~(1<<PB3);
PORTB &= ~(1<<PB0);
PORTD |= (1<<PD6);	//set E
//set the pins according to ch
if (ch & 0x01)		//lowest bit
  PORTB |= (1<<PB5);
if (ch & 0x02)		//2. bit
  PORTB |= (1<<PB4);
if (ch & 0x04)		//3. bit
  PORTB |= (1<<PB3);
if (ch & 0x08)		//4. bit
  PORTB |= (1<<PB0);
//set-> clear E
PORTD &= ~(1<<PD6);	//clear E
}

static void lcd_send(u08 ch) {
//the LCD reads the data in when switching E from high to low
//put high 4 bits on port
lcd_set_4bit(ch >> 4);
//put low 4 bits on port
lcd_set_4bit(ch);
waitms_sched(1);
}

static void lcd_send_char(u08 ch) {
PORTD |= (1<<PD5);	//set RS
lcd_send(ch);
}

static void lcd_send_sys(u08 sys) {
PORTD &= ~(1<<PD5);	//clear RS
lcd_send(sys);
waitms_sched(10);
}

#define LCD_SYMBOLS_C 16
const u08 lcd_symbols[LCD_SYMBOLS_C] PROGMEM =
      {0x00, 0x0e, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x00,  //battery symbol
       0x00, 0x06, 0x06, 0x0b, 0x0b, 0x11, 0x1f, 0x00}; //delta sembol

static void lcd_placesymbols(void) {
//starts with character 1 placing the bytes in the CG RAM
u08 syscmd = 0x48;
u08 k;
for (k = 0; k < LCD_SYMBOLS_C; k++) {
  lcd_send_sys(syscmd+k);
  u08 ch = pgm_read_byte(&lcd_symbols[k]);
  lcd_send_char(ch);
}
lcd_send_sys(0x80);	//back to DD RAM
}

static void lcd_init(void) {
waitms_sched(20);
lcd_send_sys(0x33);	//general init
lcd_send_sys(0x32);	//general init -> 4bit interface
lcd_send_sys(0x28);	//two lines and 5x7 font
lcd_send_sys(0x0c);	//display on
lcd_send_sys(0x01);	//display clear
lcd_send_sys(0x06);	//entry mode set, auto-increment address pointer
lcd_send_sys(0x02);	//return home
//place battery symbol as custom character one
lcd_placesymbols();
}

void lcd_print_word(u16 number, u08 digits) {
static u08 text[6];
/*static is simply an optimization to prevent the compiler for generating long
code (66 bytes) for making space for the string on the stack. It may be removed
if program space does not matter and must be removed if this function could be
called from multiple threads.*/
if (digits > 5)
  digits = 5;
text[digits] = '\0';
wordtostr(text, number, digits, 0);
lcd_print(text);
sched();		//because this routine was not the fastest one
}

static void lcd_put_char_virtual(u08 ch) {
lcd_pointer %= LCD_SIZE;
if (lcd_virtual[lcd_pointer] != ch) {
  lcd_virtual[lcd_pointer] = ch;	//write char
  u08 flags = lcd_modify_flags[lcd_pointer/8];	//load flags
  flags |= 1<<(lcd_pointer%8);		//set flag
  lcd_modify_flags[lcd_pointer/8] = flags;	//write back
}
lcd_pointer++;
}

void lcd_putchar(u08 ch) {
if (ch == '\n') {		//new line -> go to 2. line
  lcd_pointer = LCD_SIZE/2;
} else if (ch == '\r') {		//return to home -> go to 1. line
  lcd_pointer = 0;
} else if ((ch >= 0x80) && (ch < 0xa0)) { //set as address
  lcd_pointer = ch-0x80;
} else if (ch == '!') {		//clear lcd
  u08 k;
  for (k = 0; k < LCD_SIZE; k++) {
    lcd_put_char_virtual(' ');
  }
  lcd_pointer = 0;
} else {
  lcd_put_char_virtual(ch);
}
}

/* Puts a string from the flash into the fifo */
void lcd_print_p(PGM_VOID_P text) {
lcd_printf_p(text, 0, 0);
}

/* I came to the conclusion that using va_arg for a dynamic argument size
  produces always larger code than single calls.
  I am using the method with two parameters now because I have strings with two
  numbers relatively often.
*/
void lcd_printf_p(PGM_VOID_P text, u16 num1, u16 num2) {
u08 putchar;
u16 arg = num1;
//Count the numbers of parameters in the string
while (1) {
  //The endless loop with a break is not nice, but saves 6 bytes of flash
  putchar = pgm_read_byte(text);
  if (putchar == '\0') {
    break;
  }
  if (putchar == '$') { //we will not need this symbol on a battery charger ;-)
    text++;
    putchar = pgm_read_byte(text); //how many digits the number should have
    if (putchar == '\0') {	//just to be safe
      break;
    }
    putchar -= 48;
    lcd_print_word(arg, putchar);
    arg = num2; //switch to the second one
  } else
    lcd_putchar(putchar);
  text++;
}
}

void lcd_print(u08 *text) {
//Put data into the fifo
//in the caste the fifo is full, the data get lost
while (*(text) != '\0') {
  lcd_putchar(*(text));
  text++;
}
}

void lcd_clear(void) {
lcd_putchar('!');
}

void lcd_thread(void) {
u08 disp_wp = 0; //internal display write pointer;
lcd_init();
for (;;) {
  u08 k;
  for (k = 0; k < LCD_SIZE; k++) {
    u08 flags = lcd_modify_flags[k/8];
    if ((flags & (1<<(k%8))) != 0) {		//if char is modified
      flags &= ~(1<<(k%8));			//clear flag
      lcd_modify_flags[k/8] = flags;		//write back
      u08 ch = lcd_virtual[k];		//read out before the next sched()
      //check address
      u08 lcd_target_address = k;
      if (lcd_target_address >= (LCD_SIZE/2)) {
        lcd_target_address += (0x40-LCD_SIZE/2); //second line address
      }
      if (lcd_target_address != disp_wp) {
        lcd_send_sys(0x80 | lcd_target_address);
        disp_wp = lcd_target_address;
      }
      //write char
      lcd_send_char(ch);
      //increase address
      disp_wp++;
    }
  }
  sched();
}
}
