/*
SolarPocketLight - a simple pocket light with R + G + B + UV LED and charged by
a solar cell.
(c) 2015 by Malte Marwedel
Terms of use: GPL V2 or later
Version 1.0

1. Startup: R + G + B LED
2. Press button short: Switch LEDs off
3. Press button 0.3-1.0s: Toggle R + G + B LEDs
4. Press button 1.0-2.5s: Enable UV led + flash B LED (if not lighting)
5. Press button 2.5-4.0s: Disco mode
6. Press button for more than 12seconds: go off
7. Hold button for more than 6s on startup: go off
8. battery < 1.1V -> flash mode
9. battery < 0.8V -> go off
10. can't switch off because power button is pressed: red LED flash

PB0: output, red LED
PB1: output, green LED
PB2: output, blue LED
PB3: output, UV LED
PB4: as input: Check if key is pressed.
               Not pressed: Measure 50% of battery voltage
               Pressed: Measure approx battery voltage
     as output: Set to low, to power down the step up converter. Pressing
               the key overwrites the action of the microcontroller (resistor
               prevents short circuit)
PB5: input, measure battery voltage

You should set the brown out detector to 2.7V. BODLEVEL[1:0] = b01
The resonator to 128kHz. CLKSEL[1:0] = b11
And disable the reset pin. RSTDISBL = 0
Startup timing 14CLK + 4ms = SUT[1:0] = b01
No clock div by 8: CKDIV8 = 1
Resulting fuses:
-> High fuse: 0xFA
-> Low fuse: 0x77

With 128kHz clock, the MCU needs approx 3.34ms for one main loop -> 428 clocks
Standby current of circuit at 1.25V: 3.6µA with AVR in socket
  Maximum losses by R5 and R9: 1.2µA each (powers AVR)
Standby current of circuit at 1.25V: 3.2µA with AVR removed
  Losses by R8 + R9 series resistance: 0.6µA
  Consumption by MCP1640: ~1µA (max 2.3µA)
  Resulting unknown leakage: 0.3µA-1.6µA

Max battery standby @ 700mAh -> 22years -> battery will discharge by ~4.5% per
  as addition to the self discharge rate.
*/

#define F_CPU 128000

/* Measured pressing times:
Comfortable button press: ~200ms
Trying a short button press: ~100ms
Try a press as short as possible: ~50ms
*/
//< 300ms: go off
#define KEY_CYC_OFF 5

//< 1000ms: cycle
#define KEY_CYC_NEXT 20

//< 2500ms: UV
#define KEY_CYC_UV 50

//< 4000ms: disco mode
#define KEY_CYC_DISCO 80

//Vcc = VRef
#define REF_VOLTAGE 4.0

//good voltage >= 1.1V
#define BAT_GOOD (1023*1.1/REF_VOLTAGE)
//critical voltage: <= 0.8V
#define BAT_BAD (1023*0.8/REF_VOLTAGE)

#include <util/delay_basic.h>
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>

#if 1
ISR(TIM0_COMPA_vect, ISR_NAKED) {
//nothing ... only wakeup
	reti();
}
#else
//more overhead by the compiler
ISR(TIM0_COMPA_vect) {
//nothing ... only wakeup
}
#endif

static uint16_t getAdc(uint8_t channel) {
	PRR &= ~(1<<PRADC); //enable power to the ADC
	ADCSRA = (1<<ADEN); //div adc clock by 2 -> 64kHz conv rate
	ADCSRB = 0;
	ADMUX = channel; //Vcc as reference, no left adjust
	ADCSRA |= 1<<ADSC; //start conversion
	while(ADCSRA & 1<<ADSC); //takes 25adc cycles -> 50cpu cycles
	uint16_t value = ADC;
	ADCSRA = 0; //disable ADC
	PRR |= (1<<PRADC); //disable power to the ADC
	return value;
}

static void powerdown(void) {
	DDRB |= 0x10; //output 0
	PORTB = 1; //only red led
	while(1) {
		_delay_ms(50);
		PORTB = 0;
		_delay_ms(450);
		PORTB = 1; //only red led
	}
}

static uint8_t isKeyPressed(void) {
	uint16_t adckey = getAdc(2); //PB4
	uint16_t adcpwr = getAdc(0); //PB5
	if (adckey >= adcpwr*3/4) {
		return 1;
	}
	return 0;
}

/*
reference voltage 4V.
returns: 0: critical, 1: low, 2: good
*/
static uint8_t powerInfo(void) {
	uint16_t adc = getAdc(0); //PB5
	if (adc >= BAT_GOOD) {
		return 2;
	}
	if (adc <= BAT_BAD) {
		return 0;
	}
	return 1;
}

int main(void)
{
	uint8_t keypressedcyc = 128;
	uint8_t ledstate = 0x7; //all LEDs but UV
	uint8_t iterator = 0;
	uint8_t disco = 0;
	DDRB = 0xF; //4 led outputs
	DIDR0 = 0x3F; //disable digital input, saves power
	ACSR = 1<<ACD; //disable analog comparator, saves power
	//set timer0
	TCCR0A = 1<<WGM01; //clear timer on compare match
	TCCR0B = 1<<CS02; //divide F_CPU by 256
	OCR0A = F_CPU/256/20; //overflow every 50ms (20Hz)
	TIMSK0 = 1<<OCIE0A; //enable bit for timer0 interrupt output compare
	sei();
	for (;;) {
		uint8_t key = isKeyPressed();
		if (key) {
			keypressedcyc++;
		} else if (keypressedcyc) {
			disco = 0;
			if (keypressedcyc < KEY_CYC_OFF) {
				powerdown();
			} else if (keypressedcyc < KEY_CYC_NEXT) {
				ledstate = ((ledstate + 1) & 0x7) | (ledstate & 0x8);
				if (ledstate == 0) { //do not cycle through off
					ledstate = 1;
				}
			} else if (keypressedcyc < KEY_CYC_UV) {
				if (ledstate & 0x8) {
					ledstate &= 0x7; //UV on -> off
					if (!ledstate) {
						ledstate = 7; //if off, all LEDs on again
					}
				} else {
					ledstate |= 0x8; //UV off -> on
				}
			} else if (keypressedcyc < KEY_CYC_DISCO) {
				disco = 1;
			}
			keypressedcyc = 0;
		}
		if (disco && ((iterator & 0x3) == 0)) {
			//no UV here, as it is too dark relative to the other LEDs to be of any use
			if (!ledstate) {
				ledstate = 0x7;
			} else if (ledstate == 0x7) {
				if (iterator & 0x20) {
					ledstate = 0x1;
				} else {
					ledstate = 0x4;
				}
			} else {
				if (iterator & 0x20) {
					ledstate = (ledstate << 1) & 0x7;
				} else {
					ledstate = ledstate >> 1;
				}
			}
		}
		uint8_t powerinfo = powerInfo();
		//if constantly pressed, power down - assuming accindential pressing (eg in pocket)
		if ((powerinfo == 0) || (keypressedcyc == 255)) {
			powerdown();
		}
		iterator++;
		/* Cycle modificator (values iterator % 16):
			 1, 2, 3, 5, 6, 7, 9, 10, 11, 13, 14, 15: if power is critical -> off
			 0: if UV is enabled -> pulse blue
			 4, 8, 12: always intendet output
		*/
		if ((iterator & 0x3) && (powerinfo == 1)) {
			PORTB = 0;  //indicate low power by fast flashing
		} else if (((iterator & 0xF) == 0) && ((ledstate & 0xC) == 0x8)) { //UV without blue
			PORTB = ledstate | 0x4; //toggle blue led for indicator
			_delay_ms(0.5);
			PORTB = ledstate;
		} else {
			PORTB = ledstate;
		}
		sleep_enable();
		sleep_cpu(); //save energy
		//PORTB = 2; //test for performance and timer measurement
	}
}
