#include "main.h"
#include "clock.h"
#include "timebase.h"
#include "dcf77.h"


/*
Original Code:   Peter Dannegger
http://www.mikrocontroller.net/topic/58769

Heavily modified by: Malte Marwedel
www.marwedels.de/malte
Version 1.01 @ 2009-01-24

========== Power consumption ====================
First working version (attiny44V):
PowerOn: 577µA
During sync with 2 LEDs: 683 to 751µA
If synced: 484 µA (DCF Reciver off, Sleepmode enabled, 1MHZ, 1 LED)
  of which the LED uses 357 .. 364µA  æ 2,5Vcc
If synced (2. Multimeter): 547µA

50 mA fuse has a voltage drop of 7.7mV -> 14,07Ohm resistance (measured: 16 Ohm)

Power optimized version:
Poweron: 574µA
During sync with 2 LEDs: 690 to 751µA
If synced: 487µA

-> Does not make a difference, not using power optimized version

Used LEDs: 'LED 5-4500-RT' from Reichelt

========== Code Compilation =====================

Compiled with avr-gcc (GCC) 4.2.2 and avr-libc 1.4.7 for ATTiny44
and -O3 optimization

md5sum of main.hex: a938f91f386d58ed918b3748343dfec8
Code size: 1436 bytes

========== Program behaviour ====================
a) After power on, the Sunday and Thursday LEDs are on.
b) After some seconds, the Thursday LED shows the incoming DCF77 signal.
c) Wait until the first full time diagram is recived.
d) The LED for the current day goes on, all others are off.
e) The DCF77 reciver is turned off.
f) On 22:00 the DCF77 reciver is turned on again to synchonize. It now needs
   two correct time diagrams to sync.
g) As soon as it is synced, the program goes back into state e)
h) If the sync still did not occur within the next hour (until 23:00)
   The LED for the current day, starts to flash with a frequency of 0,5Hz

========== Pin connection =======================

PINA0: Output, Monday
PINA1: Output, Tuesday
PINA2: Output, Wednsday
PINA3: Output, Thursday
PINA4: Output, Friday
PINA5: Output, Saturday
PINA6: Output, Sunday
PINA7: Output, DCF77 On/Off
PINB0: Input with pull-up, N/C
PINB1: Input with pull-up, N/C
PINB2: Input, DCF77 non-inverting output
PINB3: N/A
PINB4: N/A
PINB5: N/A
PINB6: N/A
PINB7: N/A
*/

void display_led(u8 number) {
	number &= 0x7F;
	number |= PORTA & 0x80;
	PORTA = number;
}

void set_led(u8 lednumber) {
	if (lednumber < 7) {
		PORTA |= (1<<lednumber);
	}
}

void reset_led(u8 lednumber) {
	if (lednumber < 7) {
		PORTA &= ~(1<<lednumber);
	}
}

int main( void )
{
	//output setup
	DDRA = 0xFF;
	DDRB = 0x00;
	PORTA = 0x00;
	PORTB = 0x03;
	//set frequency and powersaving
	CLKPR = 1<<CLKPR;
	CLKPR = 1<<CLKPS1 | 1<<CLKPS0; //divider: 8 -> 1000KHZ
	//CLKPR = 1<<CLKPS2; //divider: 16 -> 500KHZ
	//CLKPR = 1<<CLKPS2 | 1<<CLKPS0; //divider: 32 -> 250KHZ
	PRR = 1<<PRTIM1 | 1<<PRUSI | 1<<PRADC; //minimize power needs
	ACSR = 1<<ACD; //disable analog comparator
	//init I/O modules
	timebase_init();
	sei();
	dcf77_on();
	display_led(0x40);
	for(;;) {
		scan_dcf77();
		if (synchronize == 0) { //only after the system start up
			if( DCF77_PIN & 1<<DCF77 ) { //let LED output dcf signal
				set_led(3);
			} else {
				reset_led(3);
			}
		} else {
			//update the LEDs
			if ((synchronize != 2) || (time.second%2)) { //flash if not synced
				//display_led(time.wday);
				u8 led = 1<<(time.wday-1);
				display_led(led);
			} else
				display_led(0);
		}
		if( timeflags & 1<<ONE_SECOND ) { //do every second
			timeflags = 0;
			clock();
			if ((synchronize == 0xff) && (time.hour == 21)) { //prepare for sync
				synchronize = 1;
			}
			if ((synchronize != 0xff) && (time.hour == 22)) { //start sync
				dcf77_on();
			}
			if (synchronize == 0xff) { //switch off as soon as its synchonized
				dcf77_off();
			}
			if ((synchronize == 1) && (time.hour == 23)) { //one hour without sync
				synchronize = 2;
			}
		}
		//enter idle mode until the timer0 wakes us up
		sleep_enable();
		sleep_cpu(); //macro from the avr-libc
		sleep_disable();
  }
}
