/* Name: main.c
 * Project: hid-mouse, a very simple HID example
 * Author: Christian Starkjohann
 * Creation Date: 2008-04-07
 * Tabsize: 4
 * Copyright: (c) 2008 by OBJECTIVE DEVELOPMENT Software GmbH
 * License: GNU GPL v2 (see License.txt), GNU GPL v3 or proprietary (CommercialLicense.txt)
 * This Revision: $Id$
 */

/*
This example should run on most AVRs with only little changes. No special
hardware resources except INT0 are used. You may have to change usbconfig.h for
different I/O pins for USB. Please note that USB D+ must be the INT0 pin, or
at least be connected to INT0 as well.

We use VID/PID 0x046D/0xC00E which is taken from a Logitech mouse. Don't
publish any hardware using these IDs! This is for demonstration only!
*/

//bootloader -d /dev/ttyS0 -b 19200 -p main.hex


#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/interrupt.h>  /* for sei() */
#include <util/delay.h>     /* for _delay_ms() */

#include <avr/pgmspace.h>   /* required by usbdrv.h */
#include <avr/eeprom.h>
#include "usbdrv.h"
//#include "oddebug.h"        /* This is also an example for using debug macros */

#include "rc5decoder.h"


static uchar    idleRate;           /* in 4 ms units */
uint8_t keyrunning;

typedef struct {
	char descr[10];
	uint8_t modifier;
	uint8_t key;
} usbkey_t;


PROGMEM char usbHidReportDescriptor[USB_CFG_HID_REPORT_DESCRIPTOR_LENGTH] = { /* USB report descriptor */
    0x05, 0x01,                    // USAGE_PAGE (Generic Desktop)
    0x09, 0x06,                    // USAGE (Keyboard)
    0xa1, 0x01,                    // COLLECTION (Application)
    0x05, 0x07,                    //   USAGE_PAGE (Keyboard)
    0x19, 0xe0,                    //   USAGE_MINIMUM (Keyboard LeftControl)
    0x29, 0xe7,                    //   USAGE_MAXIMUM (Keyboard Right GUI)
    0x15, 0x00,                    //   LOGICAL_MINIMUM (0)
    0x25, 0x01,                    //   LOGICAL_MAXIMUM (1)
    0x75, 0x01,                    //   REPORT_SIZE (1)
    0x95, 0x08,                    //   REPORT_COUNT (8)
    0x81, 0x02,                    //   INPUT (Data,Var,Abs)
    0x95, 0x01,                    //   REPORT_COUNT (1)
    0x75, 0x08,                    //   REPORT_SIZE (8)
    0x25, 0xE7,                    //   LOGICAL_MAXIMUM (231)
    0x19, 0x00,                    //   USAGE_MINIMUM (Reserved (no event indicated))
    0x29, 0xE7,                    //   USAGE_MAXIMUM (Keyboard Application)
    0x81, 0x00,                    //   INPUT (Data,Ary,Abs)
    0xc0                           // END_COLLECTION
};

#define MOD_CONTROL_LEFT    (1<<0)
#define MOD_SHIFT_LEFT      (1<<1)
#define MOD_ALT_LEFT        (1<<2)
#define MOD_GUI_LEFT        (1<<3)
#define MOD_CONTROL_RIGHT   (1<<4)
#define MOD_SHIFT_RIGHT     (1<<5)
#define MOD_ALT_RIGHT       (1<<6)
#define MOD_GUI_RIGHT       (1<<7)

#define CUSTOMKEYS 5
#define FIXEDKEYS 38
#define KEYS (FIXEDKEYS + CUSTOMKEYS)

/* Keyboard usage values, see usb.org's HID-usage-tables document, chapter
 * 10 Keyboard/Keypad Page for more codes.
 *
 * http://www.usb.org/developers/devclass_docs/Hut1_12v2.pdf
 *
 * Example for custom key: 004 043 for ALT + TAB
 *                         002 066 for Shift + F9
 *                         006 067 for Shift + Alt + F10
 *                         003 068 for Control + Alt + F11
 *                         000 118 for Menu
 */

PROGMEM const usbkey_t usbkeys[KEYS] = {
	{"0", 0, 0x27},
	{"1", 0, 0x1E},
	{"2", 0, 0x1F},
	{"3", 0, 0x20},
	{"4", 0, 0x21},
	{"5", 0, 0x22},
	{"6", 0, 0x23},
	{"7", 0, 0x24},
	{"8", 0, 0x25},
	{"9", 0, 0x26},
	{"Return", 0, 0x28},
	{"Escape", 0, 0x29},
	{"Space", 0, 0x2C},
	{"plus", 0, 0x2D},
	{"minus", 0, 0x2E},
	{"F1", 0, 0x3A},
	{"F2", 0, 0x3B},
	{"F3", 0, 0x3C},
	{"F4", 0, 0x3D},
	{"F5", 0, 0x3E},
	{"F6", 0, 0x3F},
	{"F7", 0, 0x40},
	{"F8", 0, 0x41},
	{"F9", 0, 0x42},
	{"F10", 0, 0x43},
	{"F11", 0, 0x44},
	{"F12", 0, 0x45},
	{"Pause", 0, 0x48},
	{"Page Up", 0, 0x4B},
	{"Page Down", 0, 0x4E},
	{"Cur right", 0, 0x4F},
	{"Cur left", 0, 0x50},
	{"Cur down", 0, 0x51},
	{"Cur up", 0, 0x52},
	{"Stop", 0, 0x78},
	{"Mute", 0, 0x7F},
	{"Vol up", 0, 0x80},
	{"Vol down", 0, 0x81},
	{"Custom 1", 0, 0x0}, /*Custom values will be read from eeprom*/
	{"Custom 2", 0, 0x0},
	{"Custom 3", 0, 0x0},
	{"Custom 4", 0, 0x0},
	{"Custom 5", 0, 0x0}
};

/*On my universal remote I use AUX and code 0552 */
uint32_t EEMEM eep_checksum;
uint16_t EEMEM eep_custom[CUSTOMKEYS];
uint16_t EEMEM eep_ircode[KEYS];


uchar reportBuffer[2]; /* buffer for HID reports */

/* ------------------------------------------------------------------------- */

usbMsgLen_t usbFunctionSetup(uchar data[8]) {
	usbRequest_t *rq = (void *)data;
	/* The following requests are never used. But since they are required by
	 * the specification, we implement them in this example.
	 */
	if((rq->bmRequestType & USBRQ_TYPE_MASK) == USBRQ_TYPE_CLASS) {    /* class request type */
		if(rq->bRequest == USBRQ_HID_GET_REPORT) {  /* wValue: ReportType (highbyte), ReportID (lowbyte) */
			/* we only have one report type, so don't look at wValue */
			usbMsgPtr = (void *)&reportBuffer;
			return sizeof(reportBuffer);
		} else if (rq->bRequest == USBRQ_HID_GET_IDLE) {
			usbMsgPtr = &idleRate;
			return 1;
		} else if (rq->bRequest == USBRQ_HID_SET_IDLE) {
			idleRate = rq->wValue.bytes[1];
		}
	} else {
		/* no vendor specific requests implemented */
	}
	return 0;   /* default for not implemented requests: return no data back to host */
}

static void delayms(uint16_t delay) {
	while(delay) {
		delay--;
		_delay_ms(1.0);
	}
}

static void sendusbkey(uint8_t modifier, uint8_t key) {
	reportBuffer[0] = modifier;
	reportBuffer[1] = key;
	while (!usbInterruptIsReady()) {
		usbPoll();
	}
	usbSetInterrupt((void *)&reportBuffer, sizeof(reportBuffer));
}

static void printchar(char c) {
	unsigned char modifier = 0, key = 0;
	if ((c >= 'A') && (c <= 'Z')) {
		modifier = MOD_SHIFT_LEFT;
		c = c - 'A' + 'a';
	}
	if ((c >= 'a') && (c <= 'z')) {
		key = c - 'a' + 4;
	} else if (c == ' ') {
		key = 0x2C;
	} else if (c == '.') {
		key = 0x37;
	} else if (c == '-') {
		key = 0x56;
	} else if ((c > '0') && (c <= '9')) {
		key = c - '1' + 0x1E;
	} else if (c == '0') {
		key = 0x27;
	}
	sendusbkey(modifier, key);
	//clear key
	sendusbkey(0, 0);
}

static void print(const char * text) {
	while (*text != '\0') {
		printchar(*text);
		text++;
	}
}

static void printp(PGM_P text) {
	char c;
	while(1) {
		c = pgm_read_byte(text);
		if (c == '\0')
			break;
		printchar(c);
		text++;
	}
}

//The highest number 0x17FF is 6143 = 4 digits to print
static void printnumber(uint16_t num) {
	char buffer[5];
	uint8_t i;
	for (i = 0; i < 4; i++) {
		buffer[3-i] = num % 10 + '0';
		num /= 10;
	}
	buffer[4] = '\0';
	print(buffer);
}

static void stopkey(void) {
	if (keyrunning) {
		keyrunning = 0;
		sendusbkey(0, 0);
	}
}

static uint8_t startkey(uint16_t value) {
	stopkey();
	uint8_t i, modifier, key;
	for (i = 0; i < KEYS; i++) {
		if (eeprom_read_word(eep_ircode+i) == value) {
			if (i < FIXEDKEYS) {
				modifier = pgm_read_byte(&(usbkeys[i].modifier));
				key = pgm_read_byte(&(usbkeys[i].key));
			} else {
				uint16_t val = eeprom_read_word(eep_custom + i - FIXEDKEYS);
				modifier = val >> 8;
				key = val & 0xFF;
			}
			sendusbkey(modifier, key);
			keyrunning = 1;
			return 1;
		}
	}
	return 0;
}


static uint32_t calceepromsum(void) {
	uint16_t i, val = 0;
	uint32_t checksum = 0;
	for (i = 0; i < CUSTOMKEYS; i++) {
		val = eeprom_read_word(eep_custom+i);
		checksum += (val ^ i);
	}
	for (i = 0; i < KEYS; i++) {
		val = eeprom_read_word(eep_ircode+i);
		checksum += (val ^ i);
	}
	return checksum;
}

static uint8_t checkeeprom(void) {
	uint16_t i;
	uint32_t checksum = calceepromsum();
	if (checksum != eeprom_read_dword(&eep_checksum)) {
		for (i = 0; i < CUSTOMKEYS; i++) {
			eeprom_update_word(eep_custom+i, 0);
			usbPoll();
		}
		for (i = 0; i < KEYS; i++) {
			eeprom_update_word(eep_ircode+i, 0);
			usbPoll();
		}
		return 0;
	}
	return 1;
}

static void updateeepromsum(void) {
	uint32_t checksum = calceepromsum();
	eeprom_update_dword(&eep_checksum, checksum);
}

static void setled(uint8_t state) {
	if (state) {
		PORTA &= ~(1<<PA2); //set LED
	} else {
		PORTA |= 1<<PA2; //clear LED
	}
}

static uint16_t configwaitkeypress(uint16_t lastvalue) {
	uint16_t value;
	do {
		value = rc5getdata();
		wdt_reset();
		usbPoll();
	} while ((value == 0) || (value == lastvalue));
	return value;
}

static uint8_t configreadbyte(uint16_t * val) {
	uint8_t i, j, number = 0;
	uint16_t valred;
	for (i = 0; i < 3; i++) {
		*val = configwaitkeypress(*val);
		valred = *val & 0x17FF;
		for (j = 0; j < 10; j++) {
			if (eeprom_read_word(eep_ircode + j) == valred) {
				number *= 10;
				number += j;
				printchar('0'+j);
				break;
			}
		}
		if (j == 10) {
			break; //no valid key pressed
		}
	}
	return number;
}

static void configmode(uint16_t * detectpattern, uint16_t lastval) {
	uint16_t id, i = 0;
	uint16_t val = lastval;
	uint16_t valred;
	uint8_t modifier, key;
	uint16_t ircode;
	for (;;) {
		printp(PSTR("Config mode "));
		val = configwaitkeypress(val);
		valred = val & 0x17FF;
		if (valred == detectpattern[0]) { //leave config
			printp(PSTR("Config left"));
			break;
		} else if (valred == detectpattern[1]) { //set config
			for (id = 0; id < KEYS; id++) {
				printp(usbkeys[id].descr);
				printchar(' ');
				val = configwaitkeypress(val);
				valred = val & 0x17FF;
				for (i = 0; i < id; i++) { //skip double used keys
					if (eeprom_read_word(eep_ircode+i) == valred) {
						valred = 0;
						break;
					}
				}
				printnumber(valred);
				printchar('.');
				eeprom_update_word(eep_ircode+id, valred);
			}
			updateeepromsum();
			printp(PSTR("Done "));
		} else if (valred == detectpattern[2]) { //set custom values
			for (id = 0; id < CUSTOMKEYS; id++) {
				printp(usbkeys[id+FIXEDKEYS].descr);
				printchar(' ');
				modifier = configreadbyte(&val);
				printchar(' ');
				key = configreadbyte(&val);
				uint16_t temp = (modifier << 8) | key;
				eeprom_update_word(eep_custom+id, temp);
				printchar('.');
			}
			updateeepromsum();
			printp(PSTR("Done "));
		} else if (valred == detectpattern[3]) { //print config
			for (i = 0; i < KEYS; i++) {
				ircode = eeprom_read_word(eep_ircode+i);
				if (ircode) {
					printp(usbkeys[i].descr);
					if (i >= FIXEDKEYS) {
						uint16_t temp = eeprom_read_word(eep_custom + i - FIXEDKEYS);
						modifier = temp >> 8;
						key = temp & 0xFF;
						printchar('-');
						printnumber(modifier);
						printchar('-');
						printnumber(key);
					}
					printchar(' ');
					printnumber(ircode);
					printchar('.');
					wdt_reset();
				}
			}
		} else { //just print out the keys
			printnumber(valred);
			printchar(' ');
		}
	}
	rc5getdata();
}

static uint8_t configdetect(uint16_t newval) {
	static uint16_t detectpattern[4] = {0, 0, 0};
	static uint8_t detectcnt = 0;
	//check if config wished
	if (detectcnt < 4) {
		detectpattern[detectcnt] = newval & 0x17FF;
		detectcnt++;
	} else if (detectcnt < 8) {
		if (detectpattern[detectcnt & 3] == (newval & 0x17FF)) {
			detectcnt++;
		} else {
			detectcnt = 255;
		}
		if (detectcnt == 8) {
			if ((detectpattern[0] != detectpattern[1]) &&
				  (detectpattern[0] != detectpattern[2]) &&
				  (detectpattern[0] != detectpattern[3]) &&
				  (detectpattern[1] != detectpattern[2]) &&
				  (detectpattern[1] != detectpattern[3]) &&
				  (detectpattern[2] != detectpattern[3])) {
				configmode(detectpattern, newval);
				return 1;
			}
		}
	}
	return 0;
}


int __attribute__((noreturn)) main(void) {
	uint16_t val, oldval = 0;
	uint8_t valideep;
	uint8_t isvalidkey = 0;
	CLKPR = 1 << CLKPCE;
	CLKPR = 0; //12Mhz now
	DDRA = (1<<PA1) | (1<<PA2); //LED pins as output
	DDRB = 0;
	//LED on and pull ups for IR and ununsed pins
	PORTA = (1<<PA0) | (1<<PA1) | (1<<PA3) | (1<<PA4) | (1<<PA5);
	wdt_enable(WDTO_4S);
	setled(1);
	usbInit();
	usbDeviceDisconnect();  /* enforce re-enumeration, do this while interrupts are disabled! */
	delayms(250);
	usbDeviceConnect();
	rc5Init();
	sei();
	valideep = checkeeprom();
	setled(1 - valideep);
	for(;;) {                /* main event loop */
		val = rc5getdata();
		if (val) {
			setled(valideep);
			//check if key changed
			if (val != oldval) {
				if (valideep) {
					isvalidkey = startkey(val & 0x17FF);
				}
				if (configdetect(val)) {
					valideep = checkeeprom();
				}
			}
			//start and reset disable counter
			TCCR1B = 0;
			if (isvalidkey) {
				TCNT1 = 0;
			} else {
				TCNT1 = (F_CPU/(1024*3)); //only very short LED flash on invalid key
			}
			TCCR1B = (1<<CS12) | (1<<CS10); //divide by 1024
			oldval = val;
		}
		if (TCNT1 > (F_CPU/(1024*5))) { //after 1/5s without signal
			stopkey();
			TCCR1B = 0;
			TCNT1 = 0;
			setled(1 - valideep);
		}
		wdt_reset();
		usbPoll();
	}
}
